package ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.yt;

import java.time.Instant;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.Getter;

import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.hotels.proto.TVersionList;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.AvailabilityRepository;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.CacheTransaction;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.CachedOfferAvailability;
import ru.yandex.yt.ytclient.proxy.SelectRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

public class YtAvailabilityRepository
        extends YtCacheRepository<Tuple3<String, LocalDate, LocalDate>, CachedOfferAvailability> implements AvailabilityRepository {

    @Getter
    private final String path;

    public YtAvailabilityRepository(String basePath) {
        this.path = basePath + "/offer_availability";
    }

    @Override
    protected List<Object> mapKey(Tuple3<String, LocalDate, LocalDate> id) {
        return List.of(id.get1(), id.get2().toEpochDay(), id.get3().toEpochDay());
    }

    @Override
    protected Map<String, Object> mapValue(CachedOfferAvailability item) {
        byte[] responseBytes = item.getAvailabilityResponse().toByteArray();
        byte[] versionsBytes = item.getVersions().toByteArray();
        return Map.of(
                "HotelId", item.getHotelId(),
                "CheckIn", item.getCheckinDate().toEpochDay(),
                "CheckOut", item.getCheckoutDate().toEpochDay(),
                "Response", responseBytes,
                "Versions", versionsBytes,
                "ActualizationTimestamp", item.getActualizationTimestamp().toEpochMilli());
    }

    @Override
    protected TableSchema getSchema() {
        var schemaBuilder = new TableSchema.Builder();
        schemaBuilder.addKey("HotelId", ColumnValueType.STRING);
        schemaBuilder.addKey("CheckIn", ColumnValueType.UINT64);
        schemaBuilder.addKey("CheckOut", ColumnValueType.UINT64);
        schemaBuilder.addValue("Response", ColumnValueType.STRING);
        schemaBuilder.addValue("Versions", ColumnValueType.STRING);
        schemaBuilder.addValue("ActualizationTimestamp", ColumnValueType.UINT64);
        return schemaBuilder.build();
    }

    @Override
    protected CachedOfferAvailability mapRow(UnversionedRow unversionedRow) {
        Preconditions.checkArgument(unversionedRow.getValues().size() == 6, "Unexpected number of columns");
        try {
            byte[] responseBytes = unversionedRow.getValues().get(3).bytesValue();
            byte[] versionsBytes = unversionedRow.getValues().get(4).bytesValue();
            return CachedOfferAvailability.builder()
                    .hotelId(unversionedRow.getValues().get(0).stringValue())
                    .checkinDate(LocalDate.ofEpochDay(unversionedRow.getValues().get(1).longValue()))
                    .checkoutDate(LocalDate.ofEpochDay(unversionedRow.getValues().get(2).longValue()))
                    .availabilityResponse(TJson.parseFrom(responseBytes))
                    .versions(TVersionList.parseFrom(versionsBytes))
                    .actualizationTimestamp(Instant.ofEpochMilli(unversionedRow.getValues().get(5).longValue()))
                    .build();
        } catch (InvalidProtocolBufferException e) {
            throw new RuntimeException("Unable to parse row");
        }
    }

    @Override
    public CompletableFuture<List<Tuple3<String, LocalDate, LocalDate>>> listAvailabilityKeysForOffersBeforeDate(LocalDate date, CacheTransaction transaction) {
        YtTransaction ytTransaction = (YtTransaction) transaction;
        var request = SelectRowsRequest.of(String.format("HotelId, CheckIn, CheckOut FROM [%s] WHERE CheckOut < %s",
                getPath(), date.toEpochDay()));
        return ytTransaction.getTransaction().selectRows(request)
                .thenApply(UnversionedRowset::getRows)
                .thenApply(rows -> rows.stream().map(r ->
                        Tuple3.tuple(r.getValues().get(0).stringValue(),
                                LocalDate.ofEpochDay(r.getValues().get(1).longValue()),
                                LocalDate.ofEpochDay(r.getValues().get(2).longValue())))
                        .collect(Collectors.toList()));
    }
}
