package ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.yt;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;

import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.CacheRepository;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.CacheTransaction;
import ru.yandex.yt.ytclient.proxy.LookupRowsRequest;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;

public abstract class YtCacheRepository<K, V> implements CacheRepository<K, V> {
    private final TableSchema schema = getSchema();
    private final TableSchema lookupSchema = getSchema().toLookup();


    @Override
    public CompletableFuture<V> get(K id, CacheTransaction transaction) {
        YtTransaction ytTransaction = (YtTransaction) transaction;
        return ytTransaction.getTransaction().lookupRows(prepareLookupRequest(id))
                .thenApply(rows -> {
                    Preconditions.checkState(rows.getRows().size() <= 1, "Unexpected amount of rows in lookup result");
                    if (rows.getRows().size() == 0) {
                        return null;
                    } else {
                        return mapRow(rows.getRows().get(0));
                    }
                });
    }

    @Override
    public CompletableFuture<V> put(V item, CacheTransaction transaction) {
        YtTransaction ytTransaction = (YtTransaction) transaction;
        ModifyRowsRequest request = prepareUpsertRequest(item);
        return ytTransaction.getTransaction().modifyRows(request)
                .thenApply(res -> item);
    }

    @Override
    public CompletableFuture<Void> remove(List<K> ids, CacheTransaction transaction) {
        YtTransaction ytTransaction = (YtTransaction) transaction;
        ModifyRowsRequest request = prepareRemoveRequest(ids);
        return ytTransaction.getTransaction().modifyRows(request);
    }

    protected LookupRowsRequest prepareLookupRequest(K id) {
        LookupRowsRequest request = new LookupRowsRequest(getPath(), lookupSchema);
        request.addLookupColumns(schema.getColumnNames());
        request.addFilter(mapKey(id));
        return request;
    }

    protected ModifyRowsRequest prepareUpsertRequest(V item) {
        ModifyRowsRequest request = new ModifyRowsRequest(getPath(), schema);
        Map<String, Object> values = mapValue(item);
        request.addUpdate(values);
        return request;
    }

    protected ModifyRowsRequest prepareRemoveRequest(List<K> ids) {
        ModifyRowsRequest request = new ModifyRowsRequest(getPath(), schema);
        ids.forEach(id -> request.addDelete(this.mapKey(id)));
        return request;
    }

    protected abstract List<Object> mapKey(K id);


    protected abstract Map<String, Object> mapValue(V item);

    protected abstract String getPath();

    abstract TableSchema getSchema();

    protected abstract V mapRow(UnversionedRow unversionedRow);
}
