package ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.yt;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.Getter;

import ru.yandex.travel.hotels.proto.TInventoryList;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.CacheTransaction;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.CachedHotelInventory;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.InventoryRepository;
import ru.yandex.yt.ytclient.proxy.SelectRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

public class YtInventoryRepository extends YtCacheRepository<String, CachedHotelInventory> implements InventoryRepository {

    @Getter
    private final String path;

    public YtInventoryRepository(String basePath) {
        this.path = basePath + "/inventory";
    }

    @Override
    protected List<Object> mapKey(String id) {
        return List.of(id);
    }

    @Override
    protected Map<String, Object> mapValue(CachedHotelInventory item) {
        byte[] inventoryBytes = item.getInventory().toByteArray();
        return Map.of(
                "HotelId", item.getHotelId(),
                "Version", item.getVersion(),
                "Inventory", inventoryBytes,
                "ActualizationTimestamp", item.getActualizationTimestamp().toEpochMilli()
        );
    }

    @Override
    protected TableSchema getSchema() {
        var schemaBuilder = new TableSchema.Builder();
        schemaBuilder.addKey("HotelId", ColumnValueType.STRING);
        schemaBuilder.addValue("Version", ColumnValueType.UINT64);
        schemaBuilder.addValue("Inventory", ColumnValueType.STRING);
        schemaBuilder.addValue("ActualizationTimestamp", ColumnValueType.UINT64);
        return schemaBuilder.build();
    }

    @Override
    protected CachedHotelInventory mapRow(UnversionedRow unversionedRow) {
        Preconditions.checkArgument(unversionedRow.getValues().size() == 4, "Unexpected number of columns");
        TInventoryList inventoryList = null;
        try {
            if (unversionedRow.getValues().get(2).getType() != ColumnValueType.NULL) {
                byte[] inventoryData = unversionedRow.getValues().get(2).bytesValue();
                inventoryList = TInventoryList.parseFrom(inventoryData);
            }
        } catch (InvalidProtocolBufferException e) {
            throw new RuntimeException("Unable to parse row", e);
        }
        return CachedHotelInventory.builder()
                .hotelId(unversionedRow.getValues().get(0).stringValue())
                .version(unversionedRow.getValues().get(1).longValue())
                .inventory(inventoryList)
                .actualizationTimestamp(Instant.ofEpochMilli(unversionedRow.getValues().get(3).longValue()))
                .build();
    }

    @Override
    public CompletableFuture<Map<String, Long>> getInventoryVersion(CacheTransaction transaction) {
        YtTransaction ytTransaction = (YtTransaction) transaction;
        var request = SelectRowsRequest.of(String.format("HotelId, Version FROM [%s]", getPath()));
        return ytTransaction.getTransaction().selectRows(request)
                .thenApply(UnversionedRowset::getRows)
                .thenApply(rows -> rows.stream()
                        .collect(Collectors.toMap(
                                row -> row.getValues().get(0).stringValue(),
                                row -> row.getValues().get(1).longValue())));
    }

    @Override
    public CompletableFuture<List<String>> listHotelsActualizedBefore(Instant instant, CacheTransaction transaction) {
        YtTransaction ytTransaction = (YtTransaction) transaction;
        var request = SelectRowsRequest.of(String.format("HotelId FROM [%s] WHERE ActualizationTimestamp < %s",
                getPath(), instant.toEpochMilli()));
        return ytTransaction.getTransaction().selectRows(request)
                .thenApply(UnversionedRowset::getRows)
                .thenApply(rows -> rows.stream().map(r -> r.getValues().get(0).stringValue())
                        .collect(Collectors.toList()));
    }
}
