package ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.yt;

import java.util.concurrent.CompletableFuture;

import io.micrometer.core.instrument.Metrics;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.CacheTransaction;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.ConcurrentUpdateException;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.MountInfoNotReadyException;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.rpc.RpcError;

@Slf4j
public class YtTransaction implements CacheTransaction {
    public static final int ROW_CONFLICT = 1700;
    public static final int MOUNT_INFO_NOT_READY = 1707;

    @Getter(AccessLevel.PACKAGE)
    private final ApiServiceTransaction transaction;

    public YtTransaction(ApiServiceTransaction transaction) {
        this.transaction = transaction;
        Metrics.counter("yt.l2.transactionsStarted").increment();
        log.debug("Started transaction " + transaction.getId());
    }

    @Override
    public CompletableFuture<Void> commit() {
        log.debug("Committing transaction " + transaction.getId());
        return FutureUtils.handleExceptionOfType(transaction.commit(), RpcError.class, this::handleRpcError)
                .whenComplete((r, t) -> {
                    Metrics.counter("yt.l2.transactionsFinished").increment();
                    if (t == null) {
                        log.debug("Transaction " + transaction.getId() + " committed");
                        Metrics.counter("yt.l2.transactionsCommitted").increment();
                    } else {
                        Metrics.counter("yt.l2.transactionsFailedOnCommit").increment();
                        log.error("Unable to commit transaction " + transaction.getId(), t);
                    }
                });
    }

    private <T> T handleRpcError(RpcError rpcError) {
        RpcError rowConflict = rpcError.findMatching(ROW_CONFLICT);
        RpcError mountInfoNotReady = rpcError.findMatching(MOUNT_INFO_NOT_READY);
        if (rowConflict != null) {
            throw new ConcurrentUpdateException(rowConflict);
        } else if (mountInfoNotReady != null) {
            throw new MountInfoNotReadyException(mountInfoNotReady);
        } else {
            throw rpcError;
        }
    }

    @Override
    public void close() {
        log.debug("Closing transaction " + transaction.getId());
        transaction.abort().whenComplete((r, t) -> {
            if (t == null) {
                log.debug("Transaction " + transaction.getId() + " closed");
                Metrics.counter("yt.l2.transactionsClosed").increment();
            } else {
                log.error("Unable to close transaction " + transaction.getId(), t);
                Metrics.counter("yt.l2.transactionsFailedOnClose").increment();
            }
            Metrics.counter("yt.l2.transactionsFinished").increment();
        });
    }
}
