package ru.yandex.travel.hotels.searcher.services.cache.travelline.hotels;

import java.time.Duration;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.github.benmanes.caffeine.cache.AsyncLoadingCache;
import com.github.benmanes.caffeine.cache.Caffeine;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.UpdatableDataHolder;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClient;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInfo;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelListItem;
import ru.yandex.travel.hotels.searcher.services.cache.Actualizable;
import ru.yandex.travel.hotels.searcher.services.cache.CachedActualizable;

@Slf4j
public class CachedTravellineHotelDataSearcher implements TravellineHotelDataSearcher {
    private final TravellineClient client;
    private final AsyncLoadingCache<String, HotelInfo> hotelInfoCache;
    private final Counter getCounter = Metrics.counter("searcher.partners.travelline.hotelCache.get");
    private final Counter putCounter = Metrics.counter("searcher.partners.travelline.hotelCache.put");
    private final Counter actualizeCounter = Metrics.counter("searcher.partners.travelline.hotelCache.actualize");
    private final Counter listEnabledCounter = Metrics.counter("searcher.partners.travelline.hotelCache.list");
    private final Counter disabledLoadsCounter = Metrics.counter("searcher.partners.travelline.hotelCache" +
            ".loadDisabled");
    private UpdatableDataHolder<Set<String>> enabledHotels;


    public CachedTravellineHotelDataSearcher(TravellineClient client,
                                             CachedTravellineHotelInfoSearcherProperties properties) {
        this.client = client;
        enabledHotels = new UpdatableDataHolder<>(properties.getUpdateInterval(), Duration.ofSeconds(10),
                this::getEnabledHotelsSupplier);
        hotelInfoCache = Caffeine.newBuilder()
                .maximumSize(properties.getMaxHotels())
                .expireAfterWrite(properties.getUpdateInterval())
                .buildAsync((key) -> loadHotelFromPartner("initialize-cache)", key, null));


    }

    private HotelInfo loadHotelFromPartner(String taskId, String hotelCode, String requestId) {
        var enabled = enabledHotels.get().join();
        if (enabled.contains(hotelCode)) {
            log.info("Task {}: Loading hotel info for hotel {}", taskId, hotelCode);
            putCounter.increment();
            var result = client.getHotelInfoSync(hotelCode, requestId);
            log.info("Task {}: Loaded hotel info for hotel {}", taskId, hotelCode);
            return result;
        } else {
            disabledLoadsCounter.increment();
            throw new RuntimeException("Unable to load hotel info for disabled hotel " + hotelCode);
        }
    }

    private CompletableFuture<Set<String>> getEnabledHotelsSupplier() {
        log.info("Loading a list of available hotels");
        listEnabledCounter.increment();
        return client.listHotels()
                .thenApply(hl -> hl.getHotels().stream()
                        .filter(h -> h.getInventoryVersion() > 0)
                        .map(HotelListItem::getCode)
                        .collect(Collectors.toSet()));
    }


    @Override
    public CompletableFuture<Actualizable<HotelInfo>> getHotelData(String taskId, String hotelCode,
                                                                   CallContext callContext, String requestId) {
        if (callContext.getTestContext() != null) {
            log.warn("Test context found: will skip caching");
            return client.withCallContext(callContext)
                    .getHotelInfo(hotelCode).thenApply(CachedActualizable::oneValue);
        }
        getCounter.increment();
        log.info("Task {}: getting hotelInfo for hotel {}", taskId, hotelCode);
        return hotelInfoCache
                .get(hotelCode, key -> loadHotelFromPartner(taskId, key, requestId))
                .thenApply(CachedActualizable.wrap(httpRequestId -> {
                            actualizeCounter.increment();
                            hotelInfoCache.synchronous().invalidate(hotelCode);
                            return getHotelData(taskId, hotelCode, callContext, httpRequestId);
                        }
                )).whenComplete((r, t) -> log.info("Task {}: got hotelInfo for hotel {}", taskId, hotelCode));
    }
}
