package ru.yandex.travel.hotels.searcher.yt;

import java.util.ArrayList;
import java.util.List;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.validation.Validator;

import ru.yandex.travel.commons.health.HealthCheckedSupplier;
import ru.yandex.travel.commons.messaging.CompressionSettings;
import ru.yandex.travel.commons.messaging.KeyValueStorage;
import ru.yandex.travel.commons.messaging.MessageBus;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.yt.ClientReplicatedYtProperties;
import ru.yandex.travel.commons.yt.ConnectionFactory;
import ru.yandex.travel.commons.yt.MultiClusterYtAdapter;
import ru.yandex.travel.commons.yt.SingleClusterYtAdapter;
import ru.yandex.travel.hotels.proto.TPingMessage;

@Configuration
@EnableConfigurationProperties({YtMessageBusProperties.class, YtKeyValueProperties.class, YtL2Properties.class, YtSearchFlowOfferDataStorageProperties.class})
public class YtConfiguration {
    private YtMessageBusProperties messageBusProperties;
    private YtKeyValueProperties keyValueProperties;
    private YtSearchFlowOfferDataStorageProperties searchFlowOfferDataStorageProperties;

    public YtConfiguration(YtMessageBusProperties messageBusProperties,
                           YtKeyValueProperties keyValueProperties,
                           YtSearchFlowOfferDataStorageProperties searchFlowOfferDataStorageProperties) {
        this.messageBusProperties = messageBusProperties;
        this.keyValueProperties = keyValueProperties;
        this.searchFlowOfferDataStorageProperties = searchFlowOfferDataStorageProperties;
    }

    @Bean
    public static Validator configurationPropertiesValidator() {
        return new YtPropertiesValidator();
    }

    @Bean("messageBusConnectionFactory")
    public ConnectionFactory getMessageBusFactory() {
        return new ConnectionFactory(messageBusProperties);
    }

    @Bean("keyValueStorageConnectionFactory")
    public ConnectionFactory getKeyValueStorageFactory() {
        return new ConnectionFactory(keyValueProperties);
    }

    @Bean("searchFlowOfferDataStorageConnectionFactory")
    public ConnectionFactory getSearchFlowOfferDataStorageFactory() {
        return new ConnectionFactory(searchFlowOfferDataStorageProperties);
    }

    public List<SingleClusterYtAdapter> getSingleClusterAdapters(ClientReplicatedYtProperties properties,
                                                                 String purpose,
                                                                 ConnectionFactory connectionFactory,
                                                                 boolean sorted,
                                                                 Retry retryHelper) {
        ArrayList<SingleClusterYtAdapter> result = new ArrayList<>();
        for (String clusterName : properties.getSinkClusters()) {
            result.add(new SingleClusterYtAdapter(clusterName, purpose, properties.getClusterConfigFor(clusterName),
                    new CompressionSettings(properties.getMessageCodec(), properties.getCompressionLevel()), retryHelper,
                    connectionFactory, sorted, true, false,
                    this::getPingMessage, properties.getPingLifetime(), properties.isPerInstancePingId()));
        }
        return result;
    }

    @Bean("messageBus")
    public MessageBus getMessageBus(@Qualifier("messageBusConnectionFactory") ConnectionFactory connectionFactory,
                                    Retry retryHelper) {
        MultiClusterYtAdapter adapter = new MultiClusterYtAdapter(messageBusProperties, "messageBus",
                getSingleClusterAdapters(messageBusProperties, "messageBus", connectionFactory, false, retryHelper));
        adapter.startHealthCheckThread();
        return adapter;
    }

    @Bean("keyValueStorage")
    public KeyValueStorage getKeyValueStorage(@Qualifier("keyValueStorageConnectionFactory") ConnectionFactory connectionFactory,
                                              Retry retryHelper) {
        MultiClusterYtAdapter adapter = new MultiClusterYtAdapter(keyValueProperties, "keyValueStorage",
                getSingleClusterAdapters(keyValueProperties, "keyValueStorage", connectionFactory, true, retryHelper));
        adapter.startHealthCheckThread();
        return adapter;
    }

    @Bean("searchFlowOfferDataStorage")
    public KeyValueStorage getSearchFlowOfferDataStorage(@Qualifier("searchFlowOfferDataStorageConnectionFactory") ConnectionFactory connectionFactory,
                                                         Retry retryHelper) {
        MultiClusterYtAdapter adapter = new MultiClusterYtAdapter(searchFlowOfferDataStorageProperties, "searchFlowOfferDataStorage",
                getSingleClusterAdapters(searchFlowOfferDataStorageProperties, "searchFlowOfferDataStorage", connectionFactory, true, retryHelper));
        adapter.startHealthCheckThread();
        return adapter;
    }

    @Bean("healthCheckedKeyValueStorageSupplier")
    public HealthCheckedSupplier<KeyValueStorage> storageSupplier(@Qualifier("keyValueStorage") KeyValueStorage storage) {
        return new HealthCheckedSupplier<>(storage, "hotel_offerdata_storage");
    }

    @Bean("healthCheckedSearchFlowOfferDataStorageSupplier")
    public HealthCheckedSupplier<KeyValueStorage> searchFlowOfferDataStorageSupplier(@Qualifier("searchFlowOfferDataStorage") KeyValueStorage storage) {
        return new HealthCheckedSupplier<>(storage, "search_flow_offer_data_storage");
    }

    private TPingMessage getPingMessage() {
        return TPingMessage.newBuilder().build();
    }
}
