# -*- coding: utf-8 -*-
import datetime as dt
import os
from functools import partial

from nile.api.v1 import (
    extractors as ne,
    filters as nf,
    cli,
    with_hints
)

from nile.files import LocalFile

directory = os.environ['JWD']
os.sys.path.append(directory)
from utils import PROD_DIR, make_vals, string_to_date, timestamp_to_date

ORDERS = '//home/travel/prod/cpa/hotels/orders'
FIELDS_MAP = "FieldsMap"
REDIRS = "//logs/travel-redir-log/1d"
REDIR_FIELDS = ("price", "timestamp", "operator_id", "cache_delta")
ORDER_FIELDS = ("profit_amount_rub", "order_amount_rub",
          "label_medium", "label_source",
          "label_operator_id", "label_price", "label_cache_timestamp",
          "cancelled", "nights", "order_time", "check_in", "check_out",
          'order_horizon')

make_vals_orders = with_hints(output_schema=dict(
    key=str, subkey=str, value=str))(partial(make_vals, fields=ORDER_FIELDS))


make_vals_redirs = with_hints(output_schema=dict(
    key=str, subkey=str, value=str))(partial(make_vals, fields=REDIR_FIELDS))

@cli.statinfra_job
def make_job(job, options, nirvana, statface_client):

    dates = options.dates
    date = dates[0]

    job_root = nirvana.directories[0] if nirvana.directories else PROD_DIR

    job = job.env(
        files=[LocalFile(os.path.join(directory, 'utils.py'))],
        templates=dict(job_root=job_root,
                       date=date
                       )
    )

    output_table = nirvana.output_tables[0] if nirvana.output_tables else '$job_root/to_abt/$date'

    orders = job.table(ORDERS)\
        .filter(nf.equals("has_label", True))\
        .filter(nf.custom(lambda x: timestamp_to_date(x).isoformat() == date, "created_at"))\
        .filter(nf.custom(lambda x: x, "label_serp_req_id"))\
        .filter(nf.custom(lambda x: x, "label_icookie"))\
        .project(ne.all(),
                 reqid="label_serp_req_id",
                 icookie='label_icookie',
                 cancelled=ne.custom(
                     lambda x: x != "cancelled", "status").add_hints(type=bool),
                 order_time=ne.custom(lambda x: dt.datetime.utcfromtimestamp(
                     x).isoformat(), "created_at").with_type(str),
                 order_horizon=ne.custom(lambda created, check_in:  max((string_to_date(
                     check_in) - timestamp_to_date(created)).days, 0), "created_at", "check_in").with_type(int),
                 nights=ne.custom(lambda x, y: (string_to_date(
                     y) - string_to_date(x)).days, "check_in", "check_out").with_type(int),
                 )\
        .map(make_vals_orders)


    redirs = job.table('{}/@dates'.format(REDIRS))\
        .filter(nf.custom(lambda x: bool(x["ICookie"]), FIELDS_MAP))\
        .project("timestamp",
		 label="Label",
		 cache_delta=ne.custom(lambda fields_map, ts: ts -  int(fields_map.get("CacheTimestamp")), FIELDS_MAP, "unixtime").with_type(int),
		 icookie=ne.custom(lambda x: x["ICookie"], FIELDS_MAP).add_hints(type=str),
                 reqid=ne.custom(
                     lambda x: x["SerpReqId"], FIELDS_MAP).add_hints(type=str),
                 price=ne.custom(
                     lambda x: x["Price"], FIELDS_MAP).add_hints(type=str),
                 operator_id=ne.custom(
                     lambda x: x["OperatorId"], FIELDS_MAP).add_hints(type=str),
                 )\
	.unique("label")\
        .map(make_vals_redirs)\

    job.concat(redirs, orders)\
	.sort('key', 'subkey')\
	.put(output_table)

    return job


if __name__ == "__main__":
    cli.run()
