# -*- coding: utf-8 -*-

import os
from functools import partial

from nile.api.v1 import (
    extractors as ne,
    aggregators as na,
    filters as nf,
    statface as ns,
    Record,
    cli
)

from nile.files import LocalFile

from qb2.api.v1 import (
    extractors as se,
    filters as sf,
    resources as sr
)

directory = os.environ['JWD']
os.sys.path.append(directory)
from utils import DEFAULT_DIR, TRAVEL_RUBRICS, add_totals, norm_rubric, safe_int, WizardType

REPORT_TITLE = u'Неответы геопоиска'
REPORT_PATH = "Adhoc/Hotels/Geosearch_unanswers"
REPORT_YAML_CONFIG = "unanswers/unanswers.yaml"

REQANS_FIELDS = ["reqid",
                 "origin",
                 "uid",
                 "query",
                 "rubrics",
                 "parsed_answers",
                 'raw_rubrics',
                 se.log_field('t_tot')]

MAP_REQANS = '//home/travel/fminlos/mapreqans_hotel_answers'

FIELDS = ("region", "rubric")

add_totals_curried = partial(
    add_totals, fields=FIELDS, special_cases={"rubric": 0})


@cli.statinfra_job
def make_job(job, options, nirvana, statface_client):
    """Standart function according to Statistics conventions,
    see https://clubs.at.yandex-team.ru/statistics/1143"""

    report = ns.StatfaceReport() \
        .from_yaml_config(REPORT_YAML_CONFIG)\
        .path(REPORT_PATH)\
        .title(REPORT_TITLE.encode('utf8'))\
        .scale("daily")\
        .client(statface_client)

    dates = options.dates
    if len(dates) > 1:
        suffix = "{first}_{last}".format(first=dates[0], last=dates[-1])
    else:
        suffix = dates[0]

    job_root = nirvana.directories[0] if nirvana.directories else DEFAULT_DIR

    job = job.env(
        files=[LocalFile(os.path.join(directory, 'utils.py'))],
        templates=dict(job_root=job_root,
                       suffix=suffix
                       )
    )

    output_table = nirvana.output_tables[0] if nirvana.output_tables else '$job_root/geosearch_unanswered/$suffix'

    reqans_table = '{}/@dates'.format(MAP_REQANS)

    reqans = job.table('//statbox/map-reqans-log/@dates')\
        .qb2(
        log='map-reqans-log',
        fields=REQANS_FIELDS,
        mode='yamr_lines'
    )\
        .filter(nf.custom(lambda x: len(x) > 0, "parsed_answers"))\
        .filter(nf.custom(lambda x: set(x).intersection(TRAVEL_RUBRICS), "rubrics"))\
        .filter(nf.equals("origin", "wizgeo-common-new"))\
        .project('reqid',
                 t=ne.custom(float, 't_tot'),
                 rubric=ne.custom(norm_rubric, "raw_rubrics"),
                 answer_length=ne.custom(len, "parsed_answers"))

    sessions = job.table('$job_root/user_sessions/@dates'.format(DEFAULT_DIR))\
        .filter(nf.custom(lambda blocks: sum(bool(blocks.get(wiz)) for wiz in (WizardType.Carousel, WizardType.TravelOneOrg, WizardType.OrgMn, WizardType.OneOrg)) == 0, "blocks"))\
        .filter(nf.equals("device", "desktop"))\
        .filter(nf.custom(lambda x: not x, "permalinks"))\
        .project("reqid",
                 "region",
                 fielddate=ne.custom(lambda x: x.split("T")[0], "time_isoformatted"))

    data = reqans.join(sessions, by="reqid")\
        .put('$job_root/geosearch_unanswered_data/$suffix')

    data\
        .map(add_totals_curried)\
        .groupby('fielddate', *FIELDS)\
        .aggregate(cnt=na.count(),
                   t_tot=na.median("t_tot"),
                   answer_length=na.median("answer_length"))\
        .put(output_table)\
        .publish(report)

    return job


if __name__ == "__main__":
    cli.run()
