# -*- coding: utf-8 -*-
import os
from functools import partial

from nile.api.v1 import (
    extractors as ne,
    aggregators as na,
    filters as nf,
    statface as ns,
    Record,
    cli
)

from nile.files import LocalFile

from qb2.api.v1 import (
    extractors as se,
    filters as sf,
    resources as sr
)

directory = os.environ['JWD']
os.sys.path.append(directory)
from common import DEFAULT_DIR, add_totals, norm_rubric, get_path, is_one_org, get_hotel_wizards, safe_div

REPORT_TITLE = u'Кэш-хит и кэш-мисс (new)'
REPORT_PATH = "Adhoc/Hotels/Cache-hit-again"
REPORT_YAML_CONFIG = "cache_hit_again/cache_hit.yaml"

RESULT = "/result"

SUBMIT = "/form/submit"
OFFERS = "/form/all_offers"
TABS = "company/tabs"

AT_CALENDAR = "at/calendar"
TO_CALENDAR = "to/calendar"

FIELDS = ("device", "region", "request_after_carousel_click", "rubric", "date_changed")

add_totals_curried = partial(add_totals, fields=FIELDS, special_cases={"rubric":0})

@cli.statinfra_job
def make_job(job, options, nirvana, statface_client):
    """Standart function according to Statistics conventions,
    see https://clubs.at.yandex-team.ru/statistics/1143"""

    dates = options.dates
    if len(dates) > 1:
        suffix = "{first}_{last}".format(first=dates[0], last=dates[-1])
    else:
        suffix = dates[0]

    job_root = nirvana.directories[0] if nirvana.directories else DEFAULT_DIR

    job = job.env(
        templates=dict(job_root=job_root,
	               suffix=suffix
                      )
    )

    input_table = nirvana.input_tables[0] if nirvana.input_tables else '$job_root/user_sessions/@dates'
    output_table = nirvana.output_tables[0] if nirvana.output_tables else '$job_root/cache_hit_miss_again/$suffix'

    report = ns.StatfaceReport() \
        .from_yaml_config(REPORT_YAML_CONFIG)\
        .path(REPORT_PATH)\
        .title(REPORT_TITLE.encode('utf8'))\
        .scale("daily")\
        .client(statface_client)

    job = job.env(
	files=[LocalFile(os.path.join(directory, 'common.py'))],
        templates=dict(job_root=job_root),
    )

    job.table(input_table)\
	  .filter(nf.custom(is_one_org, "blocks"))\
	  .project("region",
                   "device",
                   "permalink",
		   "blocks",
                   request_after_carousel_click=ne.const("0"),
                   fielddate=ne.custom(lambda x: x.split("T")[0], "time_isoformatted"),
	           results=ne.custom(lambda blocks: sum(elem["path"].endswith(RESULT) for elem in get_hotel_wizards(blocks) if elem.get("subtype") in ("hotel", "1org")) > 0, "blocks"),
		   was_search=ne.custom(lambda x: sum(elem["path"].endswith(SUBMIT) or elem["path"].endswith(OFFERS) for elem in x)>0, "clicks"),
 		   rubric=ne.custom(lambda x: norm_rubric(x.get("FirstRubricId")), "search_props"),
		   date_changed=ne.custom(lambda x: "yes" if sum(elem["path"].endswith(AT_CALENDAR) or elem["path"].endswith(TO_CALENDAR) for elem in x)>0 else "no", "clicks")
		)\
	  .project(ne.all(),
	           results_after_search=ne.custom(lambda blocks, was_search: was_search and sum(elem["path"].endswith(RESULT) for elem in get_hotel_wizards(blocks) if elem.get("subtype") is None) > 0, "blocks", "was_search"))\
          .map(add_totals_curried)\
	  .groupby("fielddate", *FIELDS)\
          .aggregate(cnt=na.count(),
		     results=na.sum("results"),
		     was_search=na.sum("was_search"),
		     results_after_search=na.sum("results_after_search"),
		     )\
	   .project(ne.all(),
                    cache_hit_ratio=ne.custom(safe_div, "results", "cnt"),
		    results_after_search_ratio=ne.custom(safe_div, "results_after_search", "was_search"))\
	   .put(output_table)\
           .publish(report)

    return job


if __name__ == "__main__":
    cli.run()
