# -*- coding: utf-8 -*-

import os
from functools import partial

from nile.api.v1 import (
    extractors as ne,
    aggregators as na,
    filters as nf,
    statface as ns,
    Record,
    cli
)

from nile.files import LocalFile

from qb2.api.v1 import (
    extractors as se,
    filters as sf,
    resources as sr
)

directory = os.environ['JWD']
os.sys.path.append(directory)
from common import DEFAULT_DIR, BAD_TEST_IDS, TRAVEL_RUBRICS, add_totals, norm_rubric, safe_int, WizardType

REPORT_TITLE = u'Карусель vs. MnOrg'
REPORT_PATH = "Adhoc/Hotels/Carousel-MnOrg-Bk"
REPORT_YAML_CONFIG = "carousel_vs_mnorg/carousel_mnorg.yaml"

FIELDS = ("region", "rubric", "type_of_mn")


add_totals_curried = partial(
    add_totals, fields=FIELDS, special_cases={"rubric": 0})



def safe_int(val):
   try:
      return int(val)
   except ValueError:
      return 0

def norm_rubric(rubric_id):
    return safe_int(rubric_id) if rubric_id in TRAVEL_RUBRICS else 1

def rubric_ok(rubrics):
    rubrics = {rubric for rubric in rubrics.split()}
    return len(rubrics) > 0 and (len(rubrics.intersection(TRAVEL_RUBRICS)) == len(rubrics))

def get_type(blocks):
    bk = any(elem["path"].startswith("/tabs/list") for elem in blocks.get(WizardType.Carousel, []))
    carousel =  bool(blocks.get(WizardType.Carousel)) and not bk
    org_mn = bool(blocks.get(WizardType.OrgMn))
    if carousel + org_mn + bk== 0:
	return None
    elif carousel and org_mn:
	return "carousel_org_mn"
    elif bk and org_mn:
	return "bk_org_mn"
    elif carousel and not bk:
	return "carousel"
    elif bk:
	return "bk"
    return "org_mn"

@cli.statinfra_job
def make_job(job, options, nirvana, statface_client):
    """Standart function according to Statistics conventions,
    see https://clubs.at.yandex-team.ru/statistics/1143"""

    report = ns.StatfaceReport() \
        .from_yaml_config(REPORT_YAML_CONFIG)\
        .path(REPORT_PATH)\
        .title(REPORT_TITLE.encode('utf8'))\
        .scale("daily")\
        .client(statface_client)

    dates = options.dates
    if len(dates) > 1:
        suffix = "{first}_{last}".format(first=dates[0], last=dates[-1])
    else:
        suffix = dates[0]

    job_root = nirvana.directories[0] if nirvana.directories else DEFAULT_DIR

    job = job.env(
	files=[LocalFile(os.path.join(directory, 'common.py'))],
        templates=dict(job_root=job_root,
                       suffix=suffix
                       )
    )

    input_table = nirvana.input_tables[0] if nirvana.input_tables else \
       '$job_root/user_sessions/@dates'
    output_table = nirvana.output_tables[0] if nirvana.output_tables else \
        '$job_root/carousel_mnorg/$suffix'

    data = job.table(input_table)\
	.filter(nf.custom(lambda test_ids: not BAD_TEST_IDS.intersection(test_ids), "test_ids"))\
	.filter(nf.equals("device", "desktop"))\
        .project(ne.all(),
		 type_of_mn=ne.custom(get_type, "blocks"),
                 rubrics=ne.custom(lambda search_props: search_props.get("Rubrics", ""), "search_props"))\
 	.filter(nf.custom(rubric_ok, "rubrics"))\
	.project(ne.all(),
        		 rubric=ne.custom(norm_rubric, "rubrics"))\
 	.filter(nf.not_(nf.equals("type_of_mn", None)))\
        .project("region",
                 "device",
		 "rubric",
		 "rubrics",
		 "type_of_mn",
                "query",
		"reqid",
                "test_ids",
                fielddate=ne.custom(lambda x: x.split("T")[0], "time_isoformatted"))

    data.put(output_table)

    data\
        .map(add_totals_curried)\
        .groupby("fielddate", *FIELDS)\
        .aggregate(cnt=na.count())\
        .publish(report)

    return job

if __name__ == "__main__":
    cli.run()
