# -*- coding: utf-8 -*-

from copy import copy
from itertools import combinations
from functools import partial

from nile.api.v1 import (
    extractors as ne,
    aggregators as na,
    filters as nf,
    statface as ns,
    Record,
    cli
)

from qb2.api.v1 import (
    extractors as se,
    filters as sf,
    resources as sr
)

REPORT_TITLE = u'Отельный ответ на СЕРПе (главное)'
REPORT_PATH = "Adhoc/Hotels/Serp-Main"
REPORT_YAML_CONFIG = "serp_hotels.yaml"
DEFAULT_DIR = "home/travel/analytics"

BAD_TEST_IDS = {"89647", "88844", "92282", "90255"}
FIELDS = ("device", "region", "rubric", "request_after_carousel_click", "mn_org_type", "num_of_hotel_urls", "entity_search")

TRAVEL_RUBRICS = {"184106414": u'Гостиница',
                  "184106404": u'Санаторий',
                  "184106400": u'Дом отдыха',
                  "20699506347": u'Хостел',
                  "184106426": u'Турбаза',
                  "1636761316": u'Жилье посуточно',
                  "184106420": u'Кемпинг'}


def add_totals(records, fields, special_cases):
    """The mapper should me curried before use"""
    for rec in records:
        proto_result = rec.to_dict()
        for n in range(len(fields) + 1):
            for combination in combinations(fields, n):
                result = copy(proto_result)
                for key in result.keys():
                    if key in combination:
                        if key in special_cases:
                            result[key] = special_cases[key]
                        else:
                            result[key] = "total"
                yield Record(**result)


add_totals_curried = partial(
    add_totals, fields=FIELDS, special_cases={"rubric": 0})


class WizardType(object):
    OneOrg = '1Org'
    OrgMn = 'OrgMn'
    TravelOneOrg = 'Travel1Org'
    Carousel = 'Carousel'
    OldTravel = 'HotelWizard'
    OldTours = 'ToursWizard'
    EntitySearch = 'EntitySearch'


def norm_rubric(rubric_id):
    return int(rubric_id) if rubric_id in TRAVEL_RUBRICS else 1

def get_type(blocks):
    bk = any(elem["path"].startswith("/tabs/list") for elem in blocks.get(WizardType.Carousel, []))
    carousel =  bool(blocks.get(WizardType.Carousel)) and not bk
    org_mn = bool(blocks.get(WizardType.OrgMn))
    if carousel + org_mn + bk== 0:
	return None
    elif carousel and org_mn:
	return "carousel_org_mn"
    elif bk and org_mn:
	return "bk_org_mn"
    elif carousel and not bk:
	return "carousel"
    elif bk:
	return "bk"
    return "org_mn"


@cli.statinfra_job
def make_job(job, options, nirvana, statface_client):
    """Standart function according to Statistics conventions,
    see https://clubs.at.yandex-team.ru/statistics/1143"""

    report = ns.StatfaceReport() \
        .from_yaml_config(REPORT_YAML_CONFIG)\
        .path(REPORT_PATH)\
        .title(REPORT_TITLE.encode('utf8'))\
        .scale("daily")\
        .client(statface_client)

    dates = options.dates
    if len(dates) > 1:
        suffix = "{first}_{last}".format(first=dates[0], last=dates[-1])
    else:
        suffix = dates[0]

    job_root = nirvana.directories[0] if nirvana.directories else DEFAULT_DIR

    job = job.env(
        templates=dict(job_root=job_root,
                       suffix=suffix
                       )
    )

    input_table = nirvana.input_tables[0] if nirvana.input_tables else \
       '$job_root/user_sessions/@dates'
    output_table = nirvana.output_tables[0] if nirvana.output_tables else \
        '$job_root/carousel_main_metrics/$suffix'

    job.table(input_table)\
	.filter(nf.custom(lambda test_ids: not BAD_TEST_IDS.intersection(test_ids), "test_ids"))\
        .project("region",
                 "device",
                 "search_props",
                 num_of_hotel_urls=ne.custom(lambda x: str(len(x)), "hotel_urls"),
		 mn_org_type=ne.custom(get_type, "blocks"),
                 request_after_carousel_click=ne.custom(lambda x: "after" if x else "before", "carousel_clicked_org"),
                 entity_search=ne.custom(lambda x: "yes" if x.get(
                     WizardType.EntitySearch) is not None else "no", "blocks"),
                  travel_1org=ne.custom(lambda x: x.get(
                     WizardType.TravelOneOrg) is not None, "blocks"),
                 carousel=ne.custom(lambda blocks, test_ids: blocks.get(
                     WizardType.Carousel) is not None and not BAD_TEST_IDS.intersection(test_ids), "blocks", "test_ids"),
                 travel_1org_carousel=ne.custom(lambda x: (x.get(WizardType.Carousel) is not None) and (
                     x.get(WizardType.TravelOneOrg) is not None), "blocks"),
                 vanilla_1org_carousel=ne.custom(lambda x: (x.get(WizardType.Carousel) is not None) and (
                     x.get(WizardType.OneOrg) is not None), "blocks"),
                 empty_clicked_carousel=ne.custom(lambda blocks, carousel_clicked_org: (blocks.get(WizardType.Carousel) is not None) and (blocks.get(
                     WizardType.OneOrg) is None) and (blocks.get(WizardType.TravelOneOrg) is None) and carousel_clicked_org > 0, "blocks", "carousel_clicked_org"),
                 fielddate=ne.custom(lambda x: x.split("T")[0], "time_isoformatted"))\
	.filter(sf.defined("mn_org_type"))\
        .project(ne.all(),
                 rubric=ne.custom(lambda search_props, carousel: search_props.get("Rubrics") if carousel else search_props.get("FirstRubricId"), "search_props", "carousel"))\
        .project(ne.all(),
                 rubric=ne.custom(norm_rubric, "rubric"))\
        .map(add_totals_curried)\
        .groupby("fielddate", *FIELDS)\
        .aggregate(travel_1org=na.count(predicate=nf.equals("travel_1org", True)),
                   carousel=na.count(predicate=nf.equals("carousel", True)),
                   vanilla_1org_carousel=na.count(
            predicate=nf.equals("vanilla_1org_carousel", True)),
        travel_1org_carousel=na.count(
            predicate=nf.equals("travel_1org_carousel", True)),
        empty_clicked_carousel=na.count(predicate=nf.equals("empty_clicked_carousel", True)))\
        .put(output_table)\
        .publish(report)

    return job


if __name__ == "__main__":
    cli.run()
