# -*- coding: utf-8 -*-

import os
from collections import defaultdict
from copy import copy
from itertools import combinations
from functools import partial
from os.path import abspath, join

import nile
from nile.api.v1 import (
    extractors as ne,
    aggregators as na,
    filters as nf,
    statface as ns,
    Record,
    cli
)

from nile.files import LocalFile


from qb2.api.v1 import (
    extractors as se,
    filters as sf,
    resources as sr
)

directory = os.environ['JWD']
os.sys.path.append(directory)
from common import DEFAULT_DIR,  add_totals, norm_rubric, WizardType, get_path, is_one_org

REPORT_TITLE = u'Отельный 1Org на СЕРПe (клики)'
REPORT_PATH = "Adhoc/Hotels/Travel-1Org-clicks"
REPORT_YAML_CONFIG = "travel_company/travel_one_org.yaml"
FIELDS = ("device", "region", "rubric", "click_path", "is_dynamic")


add_totals_curried = partial(
    add_totals, fields=FIELDS, special_cases={"rubric": 0})


def clicks_mapper(records):
    for rec in records:
        proto_result = dict(device=rec.device,
                            region=rec.region,
                            rubric=rec.rubric,
                            fielddate=rec.fielddate,
                            reqid=rec.reqid
                            )
        for click in rec.clicks:
            res = copy(proto_result)
            res.update(dict(
		raw_path=click["path"],
                click_path=get_path(click["path"]) or "other",
                is_dynamic=str(click["is_dynamic"]),
                dwelltime=click["dwelltime"],
                delay=click["delay"]
            ))
            yield Record(**res)


@cli.statinfra_job
def make_job(job, options, nirvana, statface_client):
    """Standart function according to Statistics conventions,
    see https://clubs.at.yandex-team.ru/statistics/1143"""

    report = ns.StatfaceReport() \
        .from_yaml_config(REPORT_YAML_CONFIG)\
        .path(REPORT_PATH)\
        .title(REPORT_TITLE.encode('utf8'))\
        .scale("daily")\
        .client(statface_client)

    dates = options.dates
    if len(dates) > 1:
        suffix = "{first}_{last}".format(first=dates[0], last=dates[-1])
    else:
        suffix = dates[0]

    job_root = nirvana.directories[0] if nirvana.directories else DEFAULT_DIR

    job = job.env(
	files=[LocalFile(os.path.join(directory, 'common.py'))],
        templates=dict(job_root=job_root,
                       suffix=suffix
                       )
    )

    input_table = nirvana.input_tables[0] if nirvana.input_tables else '$job_root/user_sessions/@dates'
    output_table = nirvana.output_tables[0] if nirvana.output_tables else '$job_root/serp_travel/$suffix'
    intermediate_table = nirvana.output_tables[0] if nirvana.output_tables else '$job_root/serp_travel/raw_data/$suffix'

    job.table(input_table)\
        .filter(nf.custom(is_one_org, "blocks"))\
        .project("region",
                 "device",
                 "search_props",
                 "clicks",
                 "reqid",
		 "permalink",
                 fielddate=ne.custom(lambda x: x.split("T")[
                     0], "time_isoformatted"))\
        .project(ne.all(),
                 rubric=ne.custom(norm_rubric, "main_rubric"),
                 )\
        .map(clicks_mapper)\
	.map(add_totals_curried)\
        .groupby("fielddate", *FIELDS)\
        .aggregate(click_count=na.count(),
                   request_count=na.count_distinct("reqid"),
                   dwelltime_median=na.median("dwelltime"),
                   delay_median=na.median("delay"))\
        .put(output_table)\
        .publish(report)

    return job


if __name__ == "__main__":
    cli.run()
