import yaml
from subprocess import run
from pathlib import Path
from argparse import ArgumentParser, Namespace
from os import uname
import getpass


txt_suggest_directory = Path('./build')
binary_suggest_directory = Path('./hotels')

hotel_suggest_builder_binary = Path('hotels_suggest_builder').resolve()
suggest_data_builder_binary = Path('suggest-data-builder').resolve()
upload_sfx_binary = Path('upload.sfx.py').resolve()


def parse_args() -> Namespace:
    parser = ArgumentParser()
    parser.add_argument('--yql-token', required=True)
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--yt-working-path')
    parser.add_argument('--skip-build', action='store_true')
    parser.add_argument('--publish', action='store_true')

    return parser.parse_args()


def build_txt_suggest(args: Namespace):
    txt_suggest_directory.mkdir(parents=True, exist_ok=True)

    run_command = [
        str(hotel_suggest_builder_binary),
        '--yql-token', args.yql_token,
        '--yt-token', args.yt_token,
        '--output-path', str(txt_suggest_directory),
    ]
    if args.yt_working_path:
        run_command.extend(['--yt-working-path', args.yt_working_path])

    run(run_command, check=True)


def build_binary_suggest():
    binary_suggest_directory.mkdir(parents=True, exist_ok=True)

    suggest_dicts = [
        "region_suggest",
        "top_hotels_suggest",
    ]
    for suggest in suggest_dicts:
        run_command = [
            str(suggest_data_builder_binary),
            '-A',
            '-r', str(txt_suggest_directory / suggest / 'ready.txt'),
            '-g', str(txt_suggest_directory / suggest / 'groups.txt'),
            '-s', str(txt_suggest_directory / suggest / 'streams.txt'),
            '-d', str(txt_suggest_directory / suggest / 'data.txt'),
            '-o', str(binary_suggest_directory / suggest),
        ]
        run(run_command, check=True)


SUGGEST_TYPE = 'hotels'
SANDBOX_TYPE = 'SUGGEST_DICT'
SANDBOX_OWNER = 'TRAVEL'
RESOURCES_FILE = 'resources.yaml'


def build_resource_and_upload_to_sandbox():
    whoami = getpass.getuser()
    hostname = uname()[1]

    resource = {
        'type': SANDBOX_TYPE,
        'description': f'Upload {SUGGEST_TYPE} from {hostname} by {whoami}',
        'path': str(binary_suggest_directory),
        'attrs': {
            'ttl': 'inf',
            'name': SUGGEST_TYPE,
        }
    }
    if args.publish:
        resource['attrs']['autodeploy'] = "yes"

    with open(RESOURCES_FILE, 'w') as file:
        yaml.dump({'resources': [resource]}, file)


if __name__ == '__main__':
    args = parse_args()

    if not args.skip_build:
        build_txt_suggest(args)
        build_binary_suggest()

    build_resource_and_upload_to_sandbox()
