import json
import logging
import sys
from argparse import ArgumentParser, Namespace
from datetime import date, datetime, timedelta, timezone
from itertools import islice
from typing import Any, Optional

from travel.library.python.currency_converter import CurrencyConverter
from travel.library.python.tools import replace_args_from_env
from travel.hotels.lib.python3.yql import yqllib
from travel.hotels.lib.python3.yt.ytlib import create_table, ensure_table_exists, ypath_join, schema_from_dict
from travel.hotels.tools.affiliate_data_builder.lib.commission_config import CommissionConfig

from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient


# touch here: 1


class AffiliateDataBuilder:
    __timezone__ = timezone(timedelta(hours=3))
    __tz_name__ = 'Europe/Moscow'

    __distribution_first_day__ = date(2021, 6, 20)

    __registered_soft_ids__ = {
        '1168',  # hotels
    }

    __distribution_report_table_schema__ = schema_from_dict({
        'soft_id': 'string',
        'clid': 'string',
    })

    __distribution_users_table_schema__ = schema_from_dict({
        'enabled_date': 'string',
        'soft_id': 'string',
        'clid': 'string',
    })

    __processed_files_table_schema__ = schema_from_dict({
        'fn': 'string',
    })

    __processed_orders_table_schema__ = schema_from_dict({
        'affiliate_id': 'string',
        'partner_name': 'string',
        'partner_order_id': 'string',
        'state': 'string',
        'currency': 'string',
        'order_amount': 'double',
        'fee_amount': 'double',
        'clid': 'string',
        'affiliate_uid': 'string',
        'created_at': 'uint64',
        'updated_at': 'uint64',
        'serviced_at': 'uint64',
        'booked_on_yandex': 'boolean',
    })

    __distribution_daily_table_schema__ = schema_from_dict({
        'billing_period': 'string',
        'product_id': 'int32',
        'clid': 'uint64',
        'currency': 'string',
        'money': 'double',
    })

    __currencies_table_schema__ = schema_from_dict({
        'currency_code': 'string',
        'rate_date': 'string',
        'rate': 'double',
    })

    def __init__(self, args: Namespace):
        self.args = args
        self.yql_client = YqlClient(db=args.yt_proxy, token=args.yql_token)
        self.yt_client = YtClient(args.yt_proxy, args.yt_token)
        self.now = datetime.now(self.__timezone__)
        self.date_today = self.now.date()

    def run(self) -> None:
        distribution_users_path = ypath_join(self.args.yt_workdir, 'distribution_users')
        distribution_users_table = ypath_join(distribution_users_path, 'merged')
        processed_orders_table = ypath_join(self.args.yt_workdir, 'processed_orders')
        output_order_index_table = ypath_join(self.args.yt_workdir, 'orders')
        output_order_dump_table = ypath_join(self.args.yt_dump_dir, 'orders')
        output_errors_table = ypath_join(self.args.yt_workdir, 'errors')
        distribution_orders_table = ypath_join(self.args.yt_workdir, 'distribution_orders')
        currencies_table = ypath_join(self.args.yt_workdir, 'currencies')
        commission_config_dump_path = ypath_join(self.args.yt_workdir, 'commission_config')
        commission_partner_config_path = ypath_join(
            commission_config_dump_path, 'latest', CommissionConfig.__partner_commission_table__
        )
        commission_user_config_path = ypath_join(
            commission_config_dump_path, 'latest', CommissionConfig.__user_commission_table__
        )

        commission_config = CommissionConfig(self.yt_client, self.__distribution_first_day__, self.now)
        commission_config.read_config(self.args.commission_config_path)
        commission_config.dump_config(commission_config_dump_path)

        with self.yt_client.Transaction() as t:
            self._update_distribution_users(t.transaction_id, distribution_users_path, distribution_users_table)

        with self.yt_client.Transaction():
            self._dump_currencies_to_yt(currencies_table)

        if not self.yt_client.exists(processed_orders_table):
            logging.info(f'Table not exists. Creating {processed_orders_table}')
            create_table(processed_orders_table, self.yt_client, self.__processed_orders_table_schema__)

        if not self.yt_client.exists(distribution_orders_table):
            logging.info(f'Table not exists. Creating {distribution_orders_table}')
            create_table(distribution_orders_table, self.yt_client, self.__processed_orders_table_schema__)

        query_args = {
            '$tz_name': self.__tz_name__,
            '$input_hotels_table': self.args.input_hotels_table,
            '$input_train_table': self.args.input_train_table,
            '$input_avia_table': self.args.input_avia_table,
            '$input_enable_train_orders': self.args.enable_train_orders,
            '$input_enable_avia_orders': self.args.enable_avia_orders,
            '$input_users_table': distribution_users_table,
            '$input_currencies_table': currencies_table,
            '$input_commission_partner_config': commission_partner_config_path,
            '$input_commission_user_config': commission_user_config_path,
            '$processed_orders_table': processed_orders_table,
            '$output_order_index_table': output_order_index_table,
            '$output_order_dump_table': output_order_dump_table,
            '$output_errors_table': output_errors_table,
            '$output_distribution_table': distribution_orders_table,
        }
        with self.yt_client.Transaction() as t:
            self._run_query('prepare_affiliate_data.yql', query_args, t.transaction_id)
            error_count = self.yt_client.row_count(output_errors_table)
            if error_count > 0:
                example = list()
                for row in islice(self.yt_client.read_table(output_errors_table), 10):
                    example.append(row)
                raise RuntimeError(f'{error_count} errors found. {example=}')

        self._fill_date_gaps(self.date_today)

        if self.yt_client.row_count(distribution_orders_table) == 0:
            logging.info('Nothing to send to distribution')
            return

        date_today = str(self.date_today)
        distribution_request_table = ypath_join(self.args.distribution_request_path, date_today)
        if self.yt_client.exists(distribution_request_table):
            logging.info(f'Table already exists {distribution_request_table}')
            return

        logging.info(f'Dumping distribution data to {distribution_request_table}')
        with self.yt_client.Transaction() as t:
            query_args = {
                '$date_today': date_today,
                '$input_orders_table': distribution_orders_table,
                '$output_aggregates_table': distribution_request_table,
            }
            self._run_query('dump_distribution_data.yql', query_args, t.transaction_id)

            logging.info(f'Removing {distribution_orders_table}')
            self.yt_client.remove(distribution_orders_table)

        logging.info('All done')

    def _update_distribution_users(
        self,
        transaction_id: str,
        distribution_users_path: str,
        distribution_users_table: str,
    ) -> None:
        logging.info(f'Updating distribution users at {distribution_users_path}')

        daily_path = ypath_join(distribution_users_path, 'daily')
        processed_files_table = ypath_join(distribution_users_path, 'processed_files')

        processed_files = self._read_processed_files(processed_files_table) - {str(self.date_today)}
        first_day = str(self.__distribution_first_day__)
        tables_to_merge = list()

        for fn in self.yt_client.list(self.args.distribution_report_path):
            fn = str(fn)
            if fn < first_day:
                continue
            if fn in processed_files:
                continue

            report_file = ypath_join(self.args.distribution_report_path, fn)
            day_table = ypath_join(daily_path, fn)
            logging.info(f'New distribution report {report_file}')

            data = list()
            for rec in json.loads(self.yt_client.read_file(report_file).read()):
                soft_id = str(rec['set.soft_id'])
                if soft_id not in self.__registered_soft_ids__:
                    continue
                data.append(dict(soft_id=soft_id, clid=str(rec['id'])))

            if self.yt_client.exists(day_table):
                logging.info(f'Removing existent table {day_table}')
                self.yt_client.remove(day_table)
            create_table(day_table, self.yt_client, self.__distribution_report_table_schema__)
            self.yt_client.write_table(day_table, data)
            tables_to_merge.append(day_table)
            processed_files.add(fn)

        if not tables_to_merge:
            logging.info('No new tables to merge')
            return

        logging.info('Merging daily tables')
        ensure_table_exists(distribution_users_table, self.yt_client, self.__distribution_users_table_schema__)
        query_args = {
            '$new_tables': tables_to_merge,
            '$result_table': distribution_users_table,
        }
        self._run_query('merge_distribution_users.yql', query_args, transaction_id)

        self._write_processed_files(processed_files, processed_files_table)
        logging.info('Distribution users update finished')

    def _dump_currencies_to_yt(self, currencies_table: str) -> None:
        logging.info('Updating currencies')
        if self.yt_client.exists(currencies_table):
            logging.info(f'Dropping existing table {currencies_table}')
            self.yt_client.remove(currencies_table)
        create_table(currencies_table, self.yt_client, self.__currencies_table_schema__)
        currency_converter = CurrencyConverter(self.args.stocks_url, self.__distribution_first_day__)
        data = list()
        for currency_code, rates in currency_converter.rates.items():
            logging.info(f'Getting {currency_code} values')
            for rate_date, rate in rates.items():
                data.append(dict(
                    currency_code=currency_code,
                    rate_date=str(rate_date),
                    rate=rate,
                ))
        curren_date = self.__distribution_first_day__
        day_delta = timedelta(days=1)
        while curren_date <= self.date_today:
            data.append(dict(
                currency_code='RUB',
                rate_date=str(curren_date),
                rate=1.,
            ))
            curren_date += day_delta
        self.yt_client.write_table(currencies_table, data)
        logging.info('Currencies updated')

    def _read_processed_files(self, path: str) -> set[str]:
        if not self.yt_client.exists(path):
            return set()
        return {rec['fn'] for rec in self.yt_client.read_table(path)}

    def _write_processed_files(self, processed_files: set[str], path: str) -> None:
        ensure_table_exists(path, self.yt_client, self.__processed_files_table_schema__)
        data = ({'fn': item} for item in processed_files)
        self.yt_client.write_table(path, data)

    def _run_query(self, query_name: str, query_args: dict[str, Any], transaction_id: Optional[str] = None) -> None:
        logging.info(f'Running query {query_name}, Args: {query_args}')
        yqllib.run_yql_file(
            client=self.yql_client,
            resource_name=query_name,
            project_name=self.__class__.__name__,
            parameters=query_args,
            transaction_id=transaction_id,
        )
        logging.info('Query finished')

    def _fill_date_gaps(self, date_today: date) -> None:
        existing_tables = set()
        if self.yt_client.exists(self.args.distribution_request_path):
            existing_tables = self.yt_client.list(self.args.distribution_request_path)
            existing_tables = set(existing_tables)
        delta = timedelta(days=1)
        current_date = date_today - delta
        while str(current_date) not in existing_tables:
            current_table = ypath_join(self.args.distribution_request_path, str(current_date))
            logging.info(f'Creating empty table {current_table}')
            create_table(current_table, self.yt_client, self.__distribution_daily_table_schema__)
            current_date -= delta

            if current_date < self.__distribution_first_day__:
                break


def main():
    logging.basicConfig(
        level=logging.INFO,
        format="%(asctime)-15s | %(module)s | %(levelname)s | %(message)s",
        stream=sys.stdout,
    )
    parser = ArgumentParser()
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--yql-token', required=True)
    parser.add_argument('--yt-workdir', required=True)
    parser.add_argument('--yt-dump-dir', required=True)
    parser.add_argument('--input-hotels-table', required=True)
    parser.add_argument('--input-train-table', required=True)
    parser.add_argument('--input-avia-table', required=True)
    parser.add_argument('--enable-train-orders', action='store_true')
    parser.add_argument('--enable-avia-orders', action='store_true')
    parser.add_argument('--distribution-report-path', default='//statbox/statbox-dict-by-name/distr_report.json')
    parser.add_argument('--distribution-request-path', required=True)
    parser.add_argument('--commission-config-path', required=True)
    parser.add_argument('--stocks-url', default='http://stocks.yandex.net/xmlhist')
    args = parser.parse_args(args=replace_args_from_env())
    AffiliateDataBuilder(args).run()


if __name__ == '__main__':
    main()
