import logging
from datetime import date
from typing import Any


ConfigItem = dict[str, Any]
ValueMapping = [dict[str, str]]


class KeyExpander:

    def __init__(self, key_fields: tuple[str, ...], allowed_values: ValueMapping):
        self.key_fields = key_fields
        self.allowed_values = allowed_values

    def get_expanded(self, item: ConfigItem) -> [ConfigItem]:
        item = {**item}
        starred_fields = list()
        for key in self.key_fields:
            if item[key] == '*':
                starred_fields.append(key)
        if not starred_fields:
            return [item]

        available_values = set()
        for row in self.allowed_values:
            values = list()
            for key in starred_fields:
                values.append(row[key])
            available_values.add(tuple(values))

        expanded_items = list()
        for values in available_values:
            item = {**item, **dict(zip(starred_fields, values))}
            expanded_items.append(item)

        return expanded_items


class OverridableConfig:

    def __init__(self, key_fields: tuple[str, ...], allowed_values: ValueMapping):
        self.key_fields = key_fields
        self.key_expander = KeyExpander(key_fields, allowed_values)

    def apply(self, raw_items: [ConfigItem], application_date: date) -> list[ConfigItem]:
        items_expanded = list()
        for item in raw_items:
            items_expanded.extend(self.key_expander.get_expanded(item))

        items_overridden = dict()
        for item in items_expanded:
            if not self.__is_applicable(item, application_date):
                logging.info(f'Not applicable {item=}')
                continue
            key = self.get_item_key(self.key_fields, item)
            items_overridden[key] = item

        return list(items_overridden.values())

    @staticmethod
    def get_item_key(key_fields: tuple[str, ...], raw_item: ConfigItem) -> tuple[str, ...]:
        key = list()
        for field_name in key_fields:
            field_value = raw_item[field_name]
            if not isinstance(field_value, str):
                raise Exception(f'Key parts should be strings but got {type(field_value)}')
            key.append(field_value)
        return tuple(key)

    @staticmethod
    def __is_applicable(raw_item: ConfigItem, current_date: date) -> bool:
        start_at = raw_item.get('StartAt')
        if start_at and current_date < date.fromisoformat(start_at):
            return False
        end_at = raw_item.get('EndAt')
        if end_at and current_date > date.fromisoformat(end_at):
            return False
        return True
