# encoding: utf-8

from gevent import monkey
monkey.patch_all()

from argparse import ArgumentParser
import gevent
import requests
import sys
import logging
from gevent.subprocess import Popen, PIPE
import os
import json
from collections import namedtuple

from travel.hotels.tools.api_req_replicator.tvm import Tvm
from travel.hotels.tools.api_req_replicator.stats import StatOwner

READ_DEPTH = 200

PATH_PREFIX_WHITELIST = {
    '/api/hotels_portal/v1': 1,
    '/api/serp/v1/get_popular_destinations': 0.001,
}

Request = namedtuple('Request', 'url headers')


class Replicator(object):

    def __init__(self, replica_url, percent, session_count, stat_owner, tvm):
        self.replica_url = replica_url
        while self.replica_url.endswith('/'):
            self.replica_url = self.replica_url[:-1]
        self.percent = percent
        self.stat_owner = stat_owner
        self.tvm = tvm
        self.free_sessions = []
        self.budget = 0
        for x in range(session_count):
            self.free_sessions.append(requests.session())

    def replicate(self, req, pct):
        self.budget += self.percent * pct
        while self.budget >= 100:
            self.budget -= 100
            if self.free_sessions:
                session = self.free_sessions.pop()
                gevent.spawn(self.replicate_glet, req, session)
                gevent.sleep(0.001)
            else:
                self.stat_owner.stat.on_no_free_session()

    def replicate_glet(self, req, session):
        full_url = self.replica_url + req.url
        headers = req.headers
        headers.pop("Host", None)
        headers.pop("X-Ya-User-Ticket", None)
        headers.pop("X-Ya-PassportId", None)
        headers.pop("X-Ya-Login", None)
        headers.update({
            'X-Ya-Service-Ticket': self.tvm.get_ticket(),
            'X-Ya-Session-Key': '00000000-0000-0000-0000-000000000000',
            'X-Ya-YandexUid': '123',
        })
        resp = session.get(full_url, headers=headers)
        if resp.status_code < 400:
            self.stat_owner.stat.on_ok()
        else:
            self.stat_owner.stat.on_http_error(resp.status_code, resp.text)
        self.free_sessions.append(session)


def read_log(filename, args, kwargs):
    while True:
        error_count = 0
        try:
            logging.debug("Opening log file '%s'" % filename)
            tail_args = ['tail', '-F', filename, '-n', str(READ_DEPTH)]
            p = Popen(tail_args, stdout=PIPE, stderr=PIPE, shell=False, close_fds=True)
            while p.returncode is None:
                line = p.stdout.readline()
                try:
                    line = line.decode('utf-8')
                    process_line(line, *args, **kwargs)
                    error_count = 0
                except Exception as e:
                    logging.error("Failed to process log line: %s" % str(e))
                    logging.error("Line: %s" % line)
                    error_count += 1
                    if error_count > 100:
                        raise Exception("Too much errors, restart")
        except Exception as e:
            logging.error("Error during log monitoring: %s" % str(e))


def start_log_mon(filename, *args, **kwargs):
    return gevent.spawn(read_log, filename, args, kwargs)


def process_line(line, replicators):
    data = json.loads(line)
    if data['method'] != 'GET':
        return
    path = data['url']
    replica_pct = None
    for pfx, pct in PATH_PREFIX_WHITELIST.items():
        if path.startswith(pfx):
            replica_pct = pct
            break
    if replica_pct is None:
        return
    if not path.startswith('/'):
        path = '/' + path
    url = path
    first = True
    for key, values in data['query_params'].items():
        for value in values:
            if first:
                url += "?"
                first = False
            else:
                url += "&"
            url += key + "=" + value
    headers = dict()
    for key, values in data['request_headers'].items():
        for value in values:
            headers[key] = value
    req = Request(url, headers)
    for r in replicators:
        r.replicate(req, replica_pct)


def main():

    parser = ArgumentParser()
    parser.add_argument('--log-file', required=True)
    parser.add_argument('--replica-url', action='append')
    parser.add_argument('--replica-percent', required=True, type=int)
    parser.add_argument('--session-count', type=int, default=100)
    parser.add_argument('--verbose', action='store_true')
    parser.add_argument("--tvm-service-id", default=2002548, type=int)
    parser.add_argument("--tvm-client-id", default=2018156, type=int)
    parser.add_argument("--tvm-client-secret", default=os.environ.get('REPLICATOR_TVM_SECRET'))

    args = parser.parse_args()
    logging.basicConfig(level=logging.DEBUG if args.verbose else logging.INFO,
                        format="%(asctime)-15s | %(module)s | %(levelname)s | %(message)s", stream=sys.stdout)

    logging.info("Replicating to %s" % args.replica_url)

    tvm = Tvm(args.tvm_service_id, args.tvm_client_id, args.tvm_client_secret)

    replicators = []
    for replica_url in args.replica_url:
        r = Replicator(replica_url, args.replica_percent, args.session_count, StatOwner(), tvm)
        replicators.append(r)
    glet = start_log_mon(args.log_file, replicators)
    glet.join()


if __name__ == '__main__':
    main()
