# coding=utf-8
from __future__ import print_function

import logging

from library.python import resource

from travel.hotels.lib.python import yqllib
from yql.api.v1.client import YqlClient
from yt.wrapper import Transaction, YtClient, write_table, read_table, create, ypath_join

from travel.hotels.lib.python.versioned_process import VersionedProcess

LOG = logging.getLogger(__name__)

FEED_TABLES = {
    'russia_apartments',
    'russia_hotels',
    'world_apartments',
    'world_hotels',
}


class CommonCatroomBuilder(VersionedProcess):
    category_name = 'catroom'
    name = 'common'

    operator_id = None  # Must be set in descendant

    DATE_FORMAT = '%Y-%m-%d'
    PERMAROOM_ID_OFFSET = 10000000
    DEFAULT_PERMAROOM_ID = 1
    HIDDEN_PERMAROOM_ID = PERMAROOM_ID_OFFSET - 1

    def __init__(self, session, args):
        super(CommonCatroomBuilder, self).__init__(session, args)

        self.yql_client = YqlClient(token=args.yql_token, db=args.yt_proxy)
        self.yt_client = YtClient(token=args.yt_token, proxy=args.yt_proxy)

        self.room_table = ypath_join(self.get_run_dir(), 'extracted_rooms')
        self.room_table_with_id = ypath_join(self.get_run_dir(), 'extracted_rooms_with_id')

        self.permalinks_table = ypath_join(self.get_run_dir(), 'permalinks')
        self.permarooms_table = ypath_join(self.get_run_dir(), 'permarooms')
        self.mappings_table = ypath_join(self.get_run_dir(), 'mappings')

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        common_group = parser.add_argument_group(CommonCatroomBuilder.name + " - common")
        common_group.add_argument("--yql-token", required=True)

    def run(self):
        self.extract_rooms()
        self.add_permaroom_id_to_rooms()
        self.build_index()

    def extract_rooms(self):
        raise NotImplementedError('To be implemented in descendent class')

    def extract_rooms_from_feed(self, feed_tables):
        self._run_query_with_log(
            '/extract_rooms_from_feed.yql',
            {
                '$operator_id': self.operator_id,
                '$feed_tables': [str(path) for path in feed_tables],
                '$target': self.room_table,
            },
        )

    def extract_rooms_from_search_result_log(self, from_day_inclusive, to_day_inclusive):
        self._run_query_with_log(
            '/extract_rooms_from_search_result_log.yql',
            {
                '$operator_id': self.operator_id,
                '$from': from_day_inclusive,
                '$to': to_day_inclusive,
                '$target': self.room_table,
            },
        )

    def add_permaroom_id_to_rooms(self):
        with Transaction(client=self.yt_client):
            def room_reader():
                current_id = self.PERMAROOM_ID_OFFSET
                for row in read_table(self.room_table, client=self.yt_client):
                    row['permaroom'] = current_id
                    current_id += 1
                    yield row

            create('table', path=self.room_table_with_id, attributes={
                'schema': [
                    {'name': 'name', 'type': 'string', },
                    {'name': 'originalId', 'type': 'string', },
                    {'name': 'originalRoomId', 'type': 'string', },
                    {'name': 'permalink', 'type': 'int64', },
                    {'name': 'permaroom', 'type': 'int64', },
                ],
            })
            write_table(self.room_table_with_id, room_reader(), client=self.yt_client)

            LOG.info('Ids successfully added to permarooms')

    def build_index(self):
        self._run_query_with_log(
            '/build_operator_catroom_index.yql',
            {
                '$operator_id': self.operator_id,
                '$source': self.room_table_with_id,
                '$permalinks_table': self.permalinks_table,
                '$permarooms_table': self.permarooms_table,
                '$mappings_table': self.mappings_table,
                '$default_permaroom_id': self.DEFAULT_PERMAROOM_ID,
                '$hidden_permaroom_id': self.HIDDEN_PERMAROOM_ID,
            },
        )

    def _run_query_with_log(self, resource_name, parameters):
        query = resource.find(resource_name).decode('utf-8')

        self._log_parameters(resource_name.lstrip('/'), parameters)

        req = yqllib.run_query(
            query=query,
            syntax_version=1,
            debug=False,
            parameters=parameters,
        )
        yqllib.wait_results(req)

    def _log_parameters(self, script_name, parameters):
        LOG.info('YQL script "{}" called with parameters \n{}'.format(
            script_name,
            '\n'.join(['{} = {};'.format(param, self._convert_value(value)) for param, value in parameters.items()]))
        )

    def _convert_value(self, value):
        if isinstance(value, str):
            return '"{}"'.format(value)
        elif isinstance(value, int):
            return '{}'.format(value)
        elif isinstance(value, list):
            return '[\n  {}\n]'.format(',\n  '.join([self._convert_value(x) for x in value]))
        else:
            return '{}'.format(value)
