package main

import (
	"bytes"
	"context"
	"encoding/gob"
	"fmt"
	"log"
	"os"
	"strconv"
	"strings"
	"time"

	tb "gopkg.in/tucnak/telebot.v2"

	"a.yandex-team.ru/travel/hotels/tools/boy_hotels_checker/internal/core"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

var actualizeButton = tb.InlineButton{
	Unique: "actualize_offers_button",
	Text:   "Актуализировать кеш",
}

var otherDatesButton = tb.InlineButton{
	Unique: "other_dates_button",
	Text:   "Проверить другую дату",
}

var otherHotelButton = tb.InlineButton{
	Unique: "other_hotel_button",
	Text:   "Проверить другой отель",
}

var checkThisHotelButton = tb.InlineButton{
	Unique: "check_this_hotel_button",
	Text:   "Проверить доступность",
}

type chatList map[int64]subscription

var partnerPrefixes = map[core.Partner][]string{
	core.Travelline: {"tl", ""},
	core.BNovo:      {"bn"},
}

var prefixToPartner = map[string]core.Partner{
	"tl": core.Travelline,
	"":   core.Travelline,
	"bn": core.BNovo,
}

type subscription struct {
	ChatID            int64
	Type              tb.ChatType
	InitiatorUsername string
	ChatName          string
	Allowed           bool
}

type TGBot struct {
	tg            *tb.Bot
	hotelsCache   core.HotelCache
	chatsToInform chatList
	channelFile   string
	ytClient      yt.Client
}

func NewBot(hotelCache *core.UpdatableHotelCache) *TGBot {
	settings := tb.Settings{
		Token:  botToken.Value,
		Poller: &tb.LongPoller{Timeout: 5 * time.Second},
	}
	tg, err := tb.NewBot(settings)
	bot := TGBot{
		tg:          tg,
		hotelsCache: hotelCache,
		channelFile: cacheDir.Value + "/subscriptions.gob",
	}
	if err != nil {
		log.Fatal(err)
	}
	hotelCache.Subscribe(bot.handleHotelChanged)
	tg.Handle("/check", bot.handleCheck)
	tg.Handle("/subscribe", bot.handleSubscribe)
	tg.Handle("/unsubscribe", bot.handleUnsubscribe)
	tg.Handle("/report", bot.handleReport)
	tg.Handle(tb.OnText, bot.handleMessage)
	tg.Handle(&actualizeButton, bot.handleActualize)
	tg.Handle(&otherDatesButton, bot.handleOtherDates)
	tg.Handle(&otherHotelButton, bot.handleOtherHotel)
	tg.Handle(&checkThisHotelButton, bot.handleThisHotel)
	tg.Handle(tb.OnQuery, bot.handleQuery)

	ytConfig := yt.Config{
		Proxy: ytProxy.Value,
		Token: ytToken.Value,
	}
	bot.ytClient, _ = ythttp.NewClient(&ytConfig)
	return &bot
}

func (bot *TGBot) StartListening() {
	err := bot.loadSubsList(context.Background())
	if err != nil {
		log.Println(fmt.Errorf("unable to load subscrptions: %w", err))
	} else {
		_ = bot.saveSubsListToFile()
	}
	log.Printf("Bot %s will now listen for incoming messages", bot.tg.Me.Username)
	bot.tg.Start()
}

func (bot *TGBot) handleQuery(query *tb.Query) {
	var results []*tb.ArticleResult

	for _, found := range bot.hotelsCache.Lookup(query.Text, 10) {
		result := &tb.ArticleResult{
			Title: fmt.Sprintf("%s (%s в %s)", found.Name, found.ID, found.Partner),
			Text:  fmt.Sprintf("/check@%s %s:%s", bot.tg.Me.Username, partnerPrefixes[found.Partner][0], found.ID)}
		results = append(results, result)
		if len(results) > 10 {
			break
		}
	}

	res := make(tb.Results, len(results))
	for i := range results {
		res[i] = results[i]
		res[i].SetResultID(strconv.Itoa(i))
	}
	err := bot.tg.Answer(query, &tb.QueryResponse{Results: res, CacheTime: 60})
	_ = err
}

func (bot *TGBot) handleHotelChanged(event core.HotelCacheUpdateEvent) {
	rm := &tb.ReplyMarkup{}
	inlineKeys := [][]tb.InlineButton{{checkThisHotelButton}}
	rm.InlineKeyboard = inlineKeys
	switch event.EventType {
	case core.NewHotel:
		log.Printf("New hotel appeared, will notify chats in 20 minutes")
		time.Sleep(time.Minute * 20)
		log.Printf("Notifying")

		for chatID, subscription := range bot.chatsToInform {
			if subscription.Allowed {
				if _, err := bot.tg.Send(&tb.Chat{ID: chatID}, fmt.Sprintf("Ура! У нас пополнение!"+
					"\nОтель %s (%s) — %s", event.Hotel.ID, event.Hotel.Partner, event.Hotel.Name), rm); err != nil {
					log.Printf("Unable to send new hotel notification: %s", err)
				}
			}
		}
	case core.RemovedHotel:
		for chatID, subscription := range bot.chatsToInform {
			if subscription.Allowed {
				if _, err := bot.tg.Send(&tb.Chat{ID: chatID},
					fmt.Sprintf("Я, возможно, заблуждаюсь, но отель %s (%s) куда-то пропал из выдачи %s. "+
						"Возможно он от них отключился :(", event.Hotel.ID, event.Hotel.Name, event.Hotel.Partner)); err != nil {
					log.Printf("Unable to send new hotel notification: %s", err)
				}
			}
		}
	}
}

func (bot *TGBot) handleSubscribe(message *tb.Message) {
	logMessage(message)
	_, exists := bot.chatsToInform[message.Chat.ID]
	if !exists {
		subscription := subscription{
			ChatID:            message.Chat.ID,
			Type:              message.Chat.Type,
			InitiatorUsername: message.Sender.Username,
			ChatName:          message.Chat.Title,
			Allowed:           true,
		}
		bot.chatsToInform[message.Chat.ID] = subscription
		err := bot.saveSubsList(context.Background())
		if err != nil {
			delete(bot.chatsToInform, message.Chat.ID)
			_, _ = bot.tg.Send(message.Chat, fmt.Errorf("не удалось изменить список подписок: %w", err))
		} else {
			_, _ = bot.tg.Send(message.Chat, "Включен информатор: теперь буду сообщать о всех подключающихся к нам новых отелях")
		}
	}
}

func (bot *TGBot) handleUnsubscribe(message *tb.Message) {
	logMessage(message)
	s, exists := bot.chatsToInform[message.Chat.ID]
	if exists && s.Allowed {
		delete(bot.chatsToInform, message.Chat.ID)
		err := bot.saveSubsList(context.Background())
		if err != nil {
			bot.chatsToInform[message.Chat.ID] = s
			_, _ = bot.tg.Send(message.Chat, fmt.Errorf("не удалось изменить список подписок: %w", err))
		} else {
			_, _ = bot.tg.Send(message.Chat, "Ок, больше не буду сообщать про новые отели")
		}
	}
}

func (bot *TGBot) handleReport(message *tb.Message) {
	logMessage(message)
	f, createdAt, err := buildReport(context.Background())
	if err != nil {
		_, _ = bot.tg.Send(message.Chat, err.Error())
		return
	}
	buffer, err := f.WriteToBuffer()
	if err != nil {
		_, _ = bot.tg.Send(message.Chat, err.Error())
		return
	}
	file := tb.File{FileReader: buffer}
	document := tb.Document{
		File:     file,
		Caption:  fmt.Sprintf("Отчет про отели, создан %s", createdAt.Format(time.RFC822)),
		FileName: fmt.Sprintf("report.%s.xlsx", strings.ReplaceAll(createdAt.Format(time.RFC3339), ":", ".")),
	}
	m, err := bot.tg.Send(message.Chat, &document)
	_ = m
	if err != nil {
		print(err)
	}
}

func (bot *TGBot) handleOtherHotel(c *tb.Callback) {
	logCallback(c, otherHotelButton.Unique)
	rm := &tb.ReplyMarkup{
		ForceReply: true,
		Selective:  true,
	}
	_, _ = bot.tg.Send(c.Message.Chat, fmt.Sprintf("@%s, %s", c.Sender.Username, hotelPrompt), &tb.SendOptions{
		ReplyMarkup: rm,
	})

	_ = bot.tg.Respond(c, &tb.CallbackResponse{})
}

func (bot *TGBot) handleThisHotel(c *tb.Callback) {
	logCallback(c, checkThisHotelButton.Unique)
	hotel, partner := parseHotelDataFromMessageTest(c.Message.Text)
	info, err := core.CheckHotel(context.Background(), bot.hotelsCache, partner, hotel, time.Time{}, time.Time{}, false)
	if err != nil {
		_, _ = bot.tg.Send(c.Message.Chat, err.Error())
	} else {
		bot.sendHotelInfo(c.Message.Chat, info)
	}
	_ = bot.tg.Respond(c, &tb.CallbackResponse{})
}

func (bot *TGBot) handleOtherDates(c *tb.Callback) {
	logCallback(c, otherDatesButton.Unique)
	hotel, partner := parseHotelDataFromMessageTest(c.Message.Text)
	if hotel != "" && partner != core.UnknownPartner {
		rm := &tb.ReplyMarkup{
			ForceReply: true,
			Selective:  true,
		}
		text := fmt.Sprintf("Отель %s (%s) — @%s, %s", hotel, partner, c.Sender.Username, datePrompt)
		_, _ = bot.tg.Send(c.Message.Chat, text, &tb.SendOptions{
			ReplyMarkup: rm,
			ReplyTo:     c.Message,
		})
	}
	_ = bot.tg.Respond(c, &tb.CallbackResponse{})
}

func (bot *TGBot) handleActualize(c *tb.Callback) {
	logCallback(c, actualizeButton.Unique)
	hotel, partner := parseHotelDataFromMessageTest(c.Message.Text)
	if hotel != "" && partner != core.UnknownPartner {
		info, err := core.CheckHotel(context.Background(), bot.hotelsCache, partner, hotel, time.Time{}, time.Time{}, true)
		if err != nil {
			_, _ = bot.tg.Send(c.Message.Chat, err.Error())
		} else {
			var templateBytes bytes.Buffer
			_ = responseTemplate.Execute(&templateBytes, info)
			rm := &tb.ReplyMarkup{}
			inlineKeys := [][]tb.InlineButton{{otherHotelButton, otherDatesButton}}
			rm.InlineKeyboard = inlineKeys
			_, _ = bot.tg.Edit(c.Message, templateBytes.String(), &tb.SendOptions{
				DisableWebPagePreview: true,
				ParseMode:             tb.ModeHTML,
				ReplyMarkup:           rm,
			})
		}
	}
	_ = bot.tg.Respond(c, &tb.CallbackResponse{})
}

func (bot *TGBot) handleMessage(message *tb.Message) {
	logMessage(message)
	if message.ReplyTo != nil {
		if strings.Contains(message.ReplyTo.Text, hotelPrompt) {
			hotel, partner := parseID(message.Text)
			if hotel != "" && partner != core.UnknownPartner {
				info, err := core.CheckHotel(context.Background(), bot.hotelsCache, partner, hotel, time.Time{}, time.Time{}, false)
				if err != nil {
					_, _ = bot.tg.Send(message.Chat, err.Error())
				} else {
					bot.sendHotelInfo(message.Chat, info)
				}
			}
		} else {
			var replyTest = message.ReplyTo.Text
			hotel, partner := parseHotelDataFromMessageTest(replyTest)
			if hotel != "" && partner != core.UnknownPartner {
				checkin, checkout := parseDates(message.Text)
				if !checkin.IsZero() {
					info, err := core.CheckHotel(context.Background(), bot.hotelsCache, partner, hotel, checkin, checkout, true)
					if err != nil {
						_, _ = bot.tg.Send(message.Chat, err.Error())
					} else {
						bot.sendHotelInfo(message.Chat, info)
					}
				}
			}
		}
	}
}

func (bot *TGBot) handleCheck(message *tb.Message) {
	logMessage(message)
	hotelID := message.Payload
	if hotelID == "" {
		rm := &tb.ReplyMarkup{
			ForceReply: true,
			Selective:  true,
		}
		_, _ = bot.tg.Send(message.Chat, hotelPrompt, &tb.SendOptions{
			ReplyTo:     message,
			ReplyMarkup: rm,
		})
	} else {
		hotel, partner := parseID(hotelID)
		if hotel != "" && partner != core.UnknownPartner {
			info, err := core.CheckHotel(context.Background(), bot.hotelsCache, partner, hotel, time.Time{}, time.Time{}, false)
			if err != nil {
				_, _ = bot.tg.Send(message.Chat, err.Error())
			} else {
				bot.sendHotelInfo(message.Chat, info)
			}
		}
	}
}

func (bot *TGBot) sendHotelInfo(chat *tb.Chat, info *core.HotelInfo) {
	var templateBytes bytes.Buffer
	_ = responseTemplate.Execute(&templateBytes, info)
	rm := &tb.ReplyMarkup{}
	if len(info.PriceChecks) > 0 && !info.PriceChecks[0].UseSearcher {
		inlineKeys := [][]tb.InlineButton{{actualizeButton}, {otherHotelButton, otherDatesButton}}
		rm.InlineKeyboard = inlineKeys
	} else {
		inlineKeys := [][]tb.InlineButton{{otherHotelButton, otherDatesButton}}
		rm.InlineKeyboard = inlineKeys
	}
	response := templateBytes.String()
	_, _ = bot.tg.Send(chat, response, &tb.SendOptions{
		DisableWebPagePreview: true,
		ParseMode:             tb.ModeHTML,
		ReplyMarkup:           rm,
	})
}

func (bot *TGBot) saveSubsList(ctx context.Context) error {
	err := bot.saveSubsListToYT(ctx)
	if err != nil {
		return fmt.Errorf("unable to save subscrptions : %w", err)
	}
	fileErr := bot.saveSubsListToFile()
	if fileErr != nil {
		println(fmt.Errorf("unable to save subscrptions to file : %w", err))
	}
	return nil
}

func (bot *TGBot) saveSubsListToFile() error {
	file, err := os.Create(bot.channelFile)
	if err != nil {
		log.Printf("Unable to save subscriptions")
		return err
	}
	defer file.Close()
	return gob.NewEncoder(file).Encode(bot.chatsToInform)
}

func (bot *TGBot) saveSubsListToYT(ctx context.Context) error {
	wctx, cancel := context.WithCancel(ctx)
	defer cancel()
	tx, err := bot.ytClient.BeginTx(wctx, nil)
	if err != nil {
		return err
	}
	path := ypath.Path(ytPath.Value + "/subscriptions")
	_, err = tx.CreateNode(wctx, path, yt.NodeTable, &yt.CreateNodeOptions{
		Recursive: true,
		Force:     true,
	})
	if err != nil {
		return err
	}
	tw, err := tx.WriteTable(wctx, path, nil)
	if err != nil {
		return err
	}
	for _, sub := range bot.chatsToInform {
		err = tw.Write(sub)
		if err != nil {
			return err
		}
	}
	err = tw.Commit()
	if err != nil {
		return err
	}
	err = tx.Commit()
	if err != nil {
		return err
	}
	return nil
}

func (bot *TGBot) loadSubsList(ctx context.Context) error {
	err := bot.loadSubsListFromYT(ctx)
	if err != nil {
		log.Println(err)
		return bot.loadSubsListFromFile()
	}
	return nil
}

func (bot *TGBot) loadSubsListFromYT(ctx context.Context) error {
	rctx, cancel := context.WithCancel(ctx)
	defer cancel()
	tx, err := bot.ytClient.BeginTx(rctx, nil)
	if err != nil {
		return fmt.Errorf("unable to start read tx: %w", err)
	}
	path := ypath.Path(ytPath.Value + "/subscriptions")
	tr, err := tx.ReadTable(rctx, path, nil)
	if err != nil {
		return fmt.Errorf("unable to read table: %w", err)
	}
	result := make(chatList)
	for tr.Next() {
		var s subscription
		err = tr.Scan(&s)
		if err != nil {
			return fmt.Errorf("unable to read item: %w", err)
		}
		result[s.ChatID] = s
	}
	bot.chatsToInform = result
	return tr.Err()
}

func (bot *TGBot) loadSubsListFromFile() error {
	result := make(chatList)
	file, err := os.Open(bot.channelFile)
	if err != nil {
		log.Printf("Unable to load subscriptions")
		bot.chatsToInform = make(chatList)
		return fmt.Errorf("unable to load subscrptions: %w", err)
	}
	defer file.Close()
	_ = gob.NewDecoder(file).Decode(&result)
	bot.chatsToInform = result
	return nil
}
