package main

import (
	"context"
	"fmt"
	"time"

	"github.com/xuri/excelize/v2"

	"a.yandex-team.ru/travel/hotels/tools/boy_hotels_checker/internal/core"
	"a.yandex-team.ru/travel/hotels/tools/boy_hotels_checker/pkg/excel"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

type reportLine struct {
	ID            string       `excel:"ID,width=5"`
	Permalink     string       `excel:"Пермалинк,width=13,link=AltayURL"`
	Partner       core.Partner `excel:"Партнер,width=8"`
	Name          string       `excel:"Название отеля,width=25,link=SiteURL"`
	Popularity    float64      `excel:"Популярность,width=12"`
	Accessible    string       `excel:"Доступен,centered,width=8"`
	AcceptedOffer string       `excel:"Оферта,centered,width=6"`
	Published     string       `excel:"Опубликован,centered,width=11"`
	Geo           string       `excel:"Гео,centered,width=4"`
	WhiteListed   string       `excel:"Белый список,centered,width=12"`
	Tomorrow      dateCheck    `excel:"Завтра на одну ночь,centered"`
	Weekend       dateCheck    `excel:"В выходные на одну ночь,centered"`
	TwoWeeks      dateCheck    `excel:"Через две недели на пять ночей,centered"`
	OneMonth      dateCheck    `excel:"Через месяц на одну ночь,centered"`
	TwoMonths     dateCheck    `excel:"Через два месяца на три ночи,centered"`
	SiteURL       string
	AltayURL      string
}

type dateCheck struct {
	Name         string `excel:"Даты,width=22,link=PortalURL"`
	HasInAPI     string `excel:"У партнера,centered,width=10"`
	HasDirect    string `excel:"У нас,centered,width=5"`
	BestPartner  string `excel:"Лучший партнер,width=14"`
	BestDirect   int    `excel:"BoY,nonzero,width=8"`
	BestClickout int    `excel:"Click-Out,nonzero,width=8"`
	BestBooking  int    `excel:"Booking,nonzero,width=8"`
	BestOstrovok int    `excel:"Ostrovok,nonzero,width=8"`
	PortalURL    string
}

func mapPriceCheck(check core.PriceCheck, hotelInfo *core.HotelInfo) dateCheck {
	res := dateCheck{}
	res.Name = fmt.Sprintf("%s — %s", check.Checkin, check.Checkout)
	if check.HasRawAPIResponses {
		res.HasInAPI = ok
	} else {
		res.HasInAPI = crit
	}
	switch check.CacheStatus {
	case core.Error:
		res.HasDirect = offerError
	case core.Miss:
		res.HasDirect = offerMiss
	case core.Empty:
		res.HasDirect = crit
	case core.Found:
		if check.Best.BestDirectPrice == check.Best.BestTotalPrice {
			res.HasDirect = ok
		} else {
			res.HasDirect = warn
		}
	}

	if check.Best.HasDirectOffers {
		res.BestDirect = check.Best.BestDirectPrice
	}
	if check.Best.HasClickOutOffers {
		res.BestClickout = check.Best.BestClickOutPrice
	}
	if check.Best.HasBookingOffers {
		res.BestBooking = check.Best.BestBookingPrice
	}
	if check.Best.HasOstrovokOffers {
		res.BestOstrovok = check.Best.BestOstrovokPrice
	}
	res.BestPartner = check.Best.BestOfferProvider
	res.PortalURL = fmt.Sprintf("https://travel.yandex.ru/hotels/permalink/%s/?adults=2&checkinDate=%s&checkoutDate=%s&childrenAges=",
		hotelInfo.Permalink, check.Checkin, check.Checkout)
	return res
}

func mapResult(hotelInfo *core.HotelInfo) reportLine {
	res := reportLine{}
	res.ID = hotelInfo.ID
	res.Partner = hotelInfo.PartnerName
	res.Permalink = hotelInfo.Permalink
	res.Popularity = hotelInfo.Popularity
	res.Name = hotelInfo.Name
	res.SiteURL = hotelInfo.URL
	res.AltayURL = fmt.Sprintf("https://altay.yandex-team.ru/cards/%s", hotelInfo.Permalink)
	if hotelInfo.IsAccessible {
		res.Accessible = ok
	} else {
		res.Accessible = crit
	}
	if hotelInfo.HasAcceptedOffer {
		res.AcceptedOffer = ok
	} else {
		res.AcceptedOffer = crit
	}
	if hotelInfo.IsPublished {
		res.Published = ok
	} else {
		res.Published = crit
	}
	if hotelInfo.HasGeo {
		res.Geo = ok
	} else {
		res.Geo = crit
	}
	if hotelInfo.IsWhiteListed {
		res.WhiteListed = ok
	} else {
		res.WhiteListed = crit
	}
	res.Tomorrow = mapPriceCheck(hotelInfo.PriceChecks[0], hotelInfo)
	res.Weekend = mapPriceCheck(hotelInfo.PriceChecks[1], hotelInfo)
	res.TwoWeeks = mapPriceCheck(hotelInfo.PriceChecks[2], hotelInfo)
	res.OneMonth = mapPriceCheck(hotelInfo.PriceChecks[3], hotelInfo)
	res.TwoMonths = mapPriceCheck(hotelInfo.PriceChecks[4], hotelInfo)
	return res
}

func buildReport(ctx context.Context) (*excelize.File, time.Time, error) {
	cfg := yt.Config{Token: ytToken.Value, Proxy: ytProxy.Value}
	client, _ := ythttp.NewClient(&cfg)
	resPath := ytPath.Value + "/results/latest"
	path := ypath.Path(resPath)
	var startedAtString string
	err := client.GetNode(ctx, ypath.Path(resPath+"/@StartedAt"), &startedAtString, nil)
	if err != nil {
		return nil, time.Time{}, err
	}
	createAt, err := time.Parse(time.RFC3339Nano, startedAtString)
	if err != nil {
		return nil, time.Time{}, err
	}
	tr, err := client.ReadTable(ctx, path, nil)
	if err != nil {
		return nil, time.Time{}, err
	}

	noDirectAtAll := excel.ReportPage{
		Title:       "Вообще без прямых офферов",
		SheetName:   "Вообще без прямых офферов",
		Description: "Отели, для которых на всех датах нет предложений на прямой продаже",
	}

	noDirectSome := excel.ReportPage{
		Title:       "Частично без прямых офферов",
		SheetName:   "Частично без прямых офферов",
		Description: "Отели, для которых на какие-либо из дат проверки есть предложения от Click-Out партнеров, но нет предложений на прямой продаже",
	}

	badPrices := excel.ReportPage{
		Title:       "Плохие цены",
		SheetName:   "Плохие цены",
		Description: "Отели, для которых на какие-либо из дат проверки предложения на прямой продаже хуже, чем цены от Click-out партера",
	}

	disconnected := excel.ReportPage{
		Title:       "Отключенные",
		SheetName:   "Отключенные",
		Description: "Отели, которые опубликованы у нас, но недоступны у партнера. Чаще всего это отели, отключившиеся от партнера",
	}

	whiteListErrors := excel.ReportPage{
		Title:       "Ошибки белого списка",
		SheetName:   "Ошибки белого списка",
		Description: "Опубликованные отели, которые не находятся в белом списке офферкеша. Такие отели — техническая ошибка кластеризации, их необходимо показать дежурному адиминистратору сервиса подключения отелей",
	}

	offercacheErrors := excel.ReportPage{
		Title:       "Ошибки Кеша",
		SheetName:   "Ошибки Кеша",
		Description: "Опубликованные отели, для которых в офферкеше после поиска не нашлось ответа. Явная техническая ошибка, подлежит расследованию.",
	}
	pages := []*excel.ReportPage{
		&noDirectAtAll, &noDirectSome, &badPrices, &disconnected, &whiteListErrors, &offercacheErrors,
	}

	for tr.Next() {
		var checkItem core.HotelInfo
		err := tr.Scan(&checkItem)
		if err != nil {
			return nil, time.Time{}, err
		}
		mapped := mapResult(&checkItem)
		if checkItem.NoDirectOffersAtAll() && checkItem.IsWhiteListed {
			noDirectAtAll.Items = append(noDirectAtAll.Items, mapped)
		}
		if checkItem.HasMissingDirectOffers() && checkItem.IsWhiteListed {
			noDirectSome.Items = append(noDirectSome.Items, mapped)
		}
		if checkItem.HasBadPrices() && checkItem.IsWhiteListed {
			badPrices.Items = append(badPrices.Items, mapped)
		}
		if checkItem.IsPublished && !checkItem.IsAccessible {
			disconnected.Items = append(disconnected.Items, mapped)
		}
		if checkItem.IsPublished && !checkItem.IsWhiteListed {
			whiteListErrors.Items = append(whiteListErrors.Items, mapped)
		}
		if checkItem.HasErrors() {
			offercacheErrors.Items = append(offercacheErrors.Items, mapped)
		}
	}

	if err := tr.Err(); err != nil {
		return nil, time.Time{}, err
	}

	f := excelize.NewFile()
	defaultSheet := f.GetSheetName(1)
	for _, p := range pages {
		if len(p.Items) == 0 {
			continue
		}
		if err := p.Render(f); err != nil {
			return nil, time.Time{}, err
		}
	}
	f.DeleteSheet(defaultSheet)
	f.SetActiveSheet(f.GetSheetIndex(noDirectSome.SheetName))
	return f, createAt, nil
}
