package main

import (
	"context"
	"fmt"
	"log"
	"time"

	"github.com/mitchellh/mapstructure"

	"a.yandex-team.ru/travel/hotels/tools/boy_hotels_checker/internal/core"
	"a.yandex-team.ru/travel/hotels/tools/boy_hotels_checker/pkg/yql"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

const query = `SELECT
  OriginalId, PartnerId, Permalink, coalesce(popularity, 0) as popularity
  FROM hahn.` + "`" + `//home/travel/prod/hotels_administrator/whitelisted_hotels` + "`" + `as wl
  JOIN hahn.` + "`" + `//home/travel/prod/general/altay_mappings/latest/permalink_to_hotel_info` + "`" + `as phi on wl.Permalink = phi.permalink
  ORDER BY popularity DESC`

type whiteListItem struct {
	OriginalID string  `mapstructure:"OriginalId"`
	PartnerID  string  `mapstructure:"PartnerId"`
	Permalink  uint64  `mapstructure:"Permalink"`
	Popularity float64 `mapstructure:"popularity"`
}

const dateLayout = "2006-01-02"

func loadWhiteList(ctx context.Context, limit int) chan *whiteListItem {
	whiteListChannel := make(chan *whiteListItem)
	go func() {
		defer func() {
			close(whiteListChannel)
			log.Println("Done loading whitelist")
		}()
		queryToRun := query
		if limit != 0 {
			queryToRun += fmt.Sprintf("\n LIMIT %d", limit)
		}

		log.Printf("Running YQL query: %s\n", queryToRun)
		operation := yql.NewOperation(queryToRun, yqlToken.Value)
		if err := operation.Start(); err != nil {
			log.Fatal(err)
		}
		if err := operation.Wait(); err != nil {
			log.Fatal(err)
		}
		log.Println("YQL query completed")

		for _, res := range operation.Results {
			var wli whiteListItem
			if err := mapstructure.Decode(res, &wli); err != nil {
				log.Fatal(err)
			}
			select {
			case whiteListChannel <- &wli:
			case <-ctx.Done():
				return
			}

		}
	}()
	return whiteListChannel
}

func checkItemsInChannel(ctx context.Context, cache core.HotelCache, itemsChan chan *whiteListItem) chan *core.HotelInfo {
	hotelInfoChannel := make(chan *core.HotelInfo)
	go func() {
		defer close(hotelInfoChannel)
		for item := range itemsChan {
			partner := core.GetPartnerByID(item.PartnerID)

			hotelInfo, err := core.CheckHotel(ctx, cache, partner, item.OriginalID, time.Time{}, time.Time{}, true)
			if err != nil {
				log.Println(fmt.Errorf("error while checking hotel %s of partner %s: %w", item.OriginalID, item.PartnerID, err))
				continue
			}
			hotelInfo.Popularity = item.Popularity

			select {
			case hotelInfoChannel <- hotelInfo:
			case <-ctx.Done():
				return
			}
		}
	}()
	return hotelInfoChannel
}

func checkRemainingItems(ctx context.Context, cache core.HotelCache) chan *core.HotelInfo {
	hotelInfoChannel := make(chan *core.HotelInfo)
	go func() {
		defer close(hotelInfoChannel)
		for _, hotel := range cache.All() {
			hotelInfo, err := core.CheckHotel(ctx, cache, hotel.Partner, hotel.ID, time.Time{}, time.Time{}, true)
			if err != nil {
				log.Println(fmt.Errorf("error while checking hotel %s of partner %s: %w", hotel.ID, hotel.Partner, err))
				continue
			}

			select {
			case hotelInfoChannel <- hotelInfo:
			case <-ctx.Done():
				return
			}
		}
	}()
	return hotelInfoChannel
}

func run(ctx context.Context, hotelCache core.HotelCache, limit int, dirPath string, sleepTime time.Duration) error {
	dateString := time.Now().Format(dateLayout)
	tablePath := fmt.Sprintf("%s/%s", dirPath, dateString)
	ctx2, cancellation := context.WithCancel(ctx)
	defer cancellation()

	whiteListChannel := loadWhiteList(ctx2, limit)

	cfg := yt.Config{Token: ytToken.Value, Proxy: ytProxy.Value}
	client, err := ythttp.NewClient(&cfg)
	if err != nil {
		return fmt.Errorf("unable to initialize yt client: %w", err)
	}
	tx, err := client.BeginTx(ctx2, nil)
	if err != nil {
		return fmt.Errorf("unable to start tx: %w", err)
	}

	path := ypath.Path(tablePath)
	loadingStarted := time.Now()
	_, err = tx.CreateNode(ctx2, path, yt.NodeTable, &yt.CreateNodeOptions{
		Force:     true,
		Recursive: true,
	})
	if err != nil {
		return fmt.Errorf("unable to initialize yt client: %w", err)
	}
	tw, err := tx.WriteTable(ctx2, path, nil)
	if err != nil {
		return fmt.Errorf("unable to create table writer: %w", err)
	}

	i := 0
	for hi := range checkItemsInChannel(ctx2, hotelCache, whiteListChannel) {
		hotelCache.Remove(hi.ID, hi.PartnerName)
		i++
		err := tw.Write(hi)
		if err != nil {
			return fmt.Errorf("unable to write %d-th result to table: %w", i, err)
		}
		time.Sleep(sleepTime)
	}

	if i < limit || limit == 0 {
		for hi := range checkRemainingItems(ctx2, hotelCache) {
			i++
			err := tw.Write(hi)
			if err != nil {
				return fmt.Errorf("unable to write %d-th result to table: %w", i, err)
			}
			if i >= limit && limit != 0 {
				break
			}
			time.Sleep(sleepTime)
		}
	}

	err = tw.Commit()
	if err != nil {
		return fmt.Errorf("unable to commit results: %w", err)
	}
	loadingFinished := time.Now()

	err = tx.SetNode(ctx2, ypath.Path(tablePath+"/@StartedAt"), loadingStarted, nil)
	if err != nil {
		return fmt.Errorf("unable to set StartedAt attribute: %w", err)
	}
	err = tx.SetNode(ctx2, ypath.Path(tablePath+"/@FinishedAt"), loadingFinished, nil)
	if err != nil {
		return fmt.Errorf("unable to set FinishedAt attribute: %w", err)
	}
	_, err = tx.LinkNode(ctx2, path, ypath.Path(dirPath+"/latest"), &yt.LinkNodeOptions{Force: true})
	if err != nil {
		return fmt.Errorf("unable to link result table to latest %w", err)
	}
	if err := tx.Commit(); err != nil {
		return fmt.Errorf("unable to commit tx: %w", err)
	}
	return nil
}
