package cache

import (
	"context"
	"strings"
)

type requestType uint

const (
	GET requestType = iota
	PUT
	LOOKUP
	REMOVE
	ALL
)

type request struct {
	requestType requestType
	keys        []interface{}
	values      []interface{}
	query       string
	limit       int
	response    chan []interface{}
}

type LookupMatcher func(value interface{}, query string) bool

type Cache struct {
	storage  map[interface{}]interface{}
	requests chan request
	matcher  LookupMatcher
	ctx      context.Context
}

func (c *Cache) Get(key interface{}) (interface{}, bool) {
	var r = request{
		requestType: GET,
		keys:        []interface{}{key},
		response:    make(chan []interface{}),
	}
	c.requests <- r
	res := <-r.response
	if res != nil {
		return res[0], true
	} else {
		return nil, false
	}
}

func (c *Cache) Put(key interface{}, value interface{}) {
	var r = request{
		requestType: PUT,
		keys:        []interface{}{key},
		values:      []interface{}{value},
	}
	c.requests <- r
}

func (c *Cache) PutAll(keys []interface{}, values []interface{}) {
	var r = request{
		requestType: PUT,
		keys:        keys,
		values:      values,
	}
	c.requests <- r
}
func (c *Cache) Lookup(query string, limit int) []interface{} {
	var r = request{
		requestType: LOOKUP,
		limit:       limit,
		query:       strings.ToLower(query),
		response:    make(chan []interface{}),
	}
	c.requests <- r
	res := <-r.response
	return res
}

func (c *Cache) Remove(key interface{}) {
	var r = request{
		requestType: REMOVE,
		keys:        []interface{}{key},
	}
	c.requests <- r
}

func (c *Cache) All() []interface{} {
	var r = request{
		requestType: ALL,
		response:    make(chan []interface{}),
	}
	c.requests <- r
	res := <-r.response
	return res
}

func (c *Cache) run() {
	for {
		select {
		case req := <-c.requests:
			switch req.requestType {
			case GET:
				val, exists := c.storage[req.keys[0]]
				var result []interface{} = nil
				if exists {
					result = make([]interface{}, 1)
					result[0] = val
				}
				select {
				case req.response <- result:
					close(req.response)
					continue
				case <-c.ctx.Done():
					return
				}
			case REMOVE:
				delete(c.storage, req.keys[0])
			case PUT:
				if len(req.keys) != len(req.values) {
					panic("Keys and values do not match")
				}
				for i := range req.keys {
					c.storage[req.keys[i]] = req.values[i]
				}
			case LOOKUP:
				var result []interface{}
				if c.matcher != nil {
					for _, item := range c.storage {
						if c.matcher(item, req.query) {
							result = append(result, item)
							if len(result) >= req.limit {
								break
							}
						}
					}
				}
				select {
				case req.response <- result:
					close(req.response)
					continue
				case <-c.ctx.Done():
					return
				}
			case ALL:
				var result = make([]interface{}, len(c.storage))
				i := 0
				for _, v := range c.storage {
					result[i] = v
					i++
				}
				select {
				case req.response <- result:
					close(req.response)
					continue
				case <-c.ctx.Done():
					return
				}
			}
		case <-c.ctx.Done():
			return
		}
	}
}

func NewCache(ctx context.Context, matcher LookupMatcher) Cache {
	var cache = Cache{
		storage:  make(map[interface{}]interface{}),
		requests: make(chan request),
		matcher:  matcher,
		ctx:      ctx,
	}
	go cache.run()
	return cache
}
