package core

import (
	"a.yandex-team.ru/travel/hotels/tools/boy_hotels_checker/pkg/utils"
	"fmt"
	"log"
	"net/http"
	"net/url"
	"strconv"
	"time"
)

var bnovoToken string
var tokenUpdatedAt time.Time

type bNovoRequest struct {
	Accounts    []int  `json:"accounts"`
	CheckinFrom string `json:"checkin_from"`
	Nights      int    `json:"nights"`
}

func checkBnovoAcceptance(hotelID string) (bool, error) {
	token, err := getBnovoToken()
	if err != nil {
		return false, fmt.Errorf("unable to check bnovo acceptance: %w", err)
	}
	args := url.Values{}
	args.Add("token", token)
	args.Add("account_id", hotelID)
	parsed, err := utils.HTTPGet(bnovoPrivateAPI.Value+"/yandex_hotel_details", args, http.Header{})
	if err != nil {
		httpErr, matches := err.(utils.UnexpectedHTTPStatusError)
		if matches && (httpErr.StatusCode == 403 || httpErr.StatusCode == 404) {
			return false, nil
		}
		return false, fmt.Errorf("unable to check bnovo acceptance: %w", err)
	}
	if parsed.Exists("hotel_details", "bank_account_details", "inn") {
		inn := string(parsed.GetStringBytes("hotel_details", "bank_account_details", "inn"))
		return checkAgreementStatus(BNovo, inn)
	}
	return false, nil
}

func bnovoHotelUpdater(cache HotelCache) ([]HotelCacheUpdateEvent, error) {
	var result []HotelCacheUpdateEvent
	token, err := getBnovoToken()
	if err != nil {
		return nil, fmt.Errorf("unable to list bnovo hotels: %w", err)
	}
	args := url.Values{}
	args.Add("token", token)
	parsed, err := utils.HTTPGet(bnovoPrivateAPI.Value+"/accounts", args, http.Header{})
	if err != nil {
		return nil, fmt.Errorf("unable to list bnovo hotels: %w", err)
	}
	existingCopy := map[string]*HotelCacheItem{}
	for _, v := range cache.All() {
		if v.Partner == BNovo {
			existingCopy[v.ID] = v
		}
	}
	if parsed.Exists("accounts") {
		for _, account := range parsed.GetArray("accounts") {
			name := string(account.GetStringBytes("name"))
			id := strconv.Itoa(account.GetInt("id"))
			found := cache.Get(id, BNovo)
			if found == nil {
				log.Printf("New Bnovo hotel with id '%s' and name '%s'", id, name)
				item := &HotelCacheItem{
					ID:      id,
					Name:    name,
					Partner: BNovo,
				}
				cache.Put(item)
				r := HotelCacheUpdateEvent{
					EventType: NewHotel,
					Hotel:     item,
				}
				result = append(result, r)
			}
			delete(existingCopy, id)
		}
		for id, item := range existingCopy {
			log.Printf("Removing Bnovo hotel with id '%s' and name '%s'", id, item.Name)
			cache.Remove(id, BNovo)
			r := HotelCacheUpdateEvent{
				EventType: RemovedHotel,
				Hotel:     item,
			}
			result = append(result, r)
		}
	}
	return result, nil
}

func checkBnovoAvailability(hotelID string, check *PriceCheck) error {
	hotelIDInt, _ := strconv.Atoi(hotelID)
	request := bNovoRequest{
		Accounts:    []int{hotelIDInt},
		CheckinFrom: check.Checkin,
		Nights:      check.Nights,
	}
	parsed, err := utils.HTTPPostJSON(bnovoPublicAPI.Value+"/prices_los", request, http.Header{})
	if err != nil {
		unexpectedStatus, ok := err.(utils.UnexpectedHTTPStatusError)
		if ok && unexpectedStatus.StatusCode == 404 {
			return nil
		} else {
			return fmt.Errorf("unable to check for bnovo availability: %w", err)
		}
	}
	if parsed.Exists(hotelID, check.Checkin, "rates") {
		amount := len(parsed.GetArray(hotelID, check.Checkin, "rates"))
		check.HasRawAPIResponses = amount > 0
		return nil
	}
	return nil
}

func getBnovoToken() (string, error) {
	if bnovoToken == "" || tokenUpdatedAt.Before(time.Now().Add(-1*time.Hour)) {
		newToken, err := authorize()
		if err != nil {
			return "", fmt.Errorf("authorization failed: %w", err)
		}
		bnovoToken = newToken
		tokenUpdatedAt = time.Now()
		return bnovoToken, nil
	} else {
		return bnovoToken, nil
	}
}

func authorize() (string, error) {
	data := url.Values{}
	data.Add("username", bnovoUsername.Value)
	data.Add("password", bnovoPassword.Value)
	targetURL := bnovoPrivateAPI.Value + "/auth"
	parsed, err := utils.HTTPPostForm(targetURL, data)
	if err != nil {
		return "", fmt.Errorf("unable to authorize: %w", err)
	}
	return string(parsed.GetStringBytes("token")), nil
}
