package core

// Here we have main checking logic: code to list hotels in partners' APIs, inspect their publishing status
// in geo, ext-api, whitelist etc, as well as lookup for offer availability in offer cache

import (
	hproto "a.yandex-team.ru/travel/hotels/proto2"
	tproto "a.yandex-team.ru/travel/proto"
	"context"
	"errors"
	"fmt"
	"log"
	"math"
	"strings"
	"time"
)

const dateLayout = "2006-01-02"

type Partner string

var Travelline Partner = "travelline"
var BNovo Partner = "bnovo"
var UnknownPartner Partner

var partnerHotelUpdaters = map[Partner]HotelUpdater{
	Travelline: travellineHotelUpdater,
	BNovo:      bnovoHotelUpdater,
}

type CacheStatus string

const (
	Miss       CacheStatus = "HCS_Miss"
	Found      CacheStatus = "HCS_Found"
	Empty      CacheStatus = "HCS_Empty"
	Error      CacheStatus = "HCS_Error"
	Restricted CacheStatus = "HCS_RestrictedBySearchSubKey"
)

type HotelInfo struct {
	ID               string
	PartnerName      Partner
	PartnerID        string
	URL              string
	HasAcceptedOffer bool
	IsAccessible     bool
	IsPublished      bool
	HasGeo           bool
	Permalink        string
	Name             string
	NameByPartner    string
	IsWhiteListed    bool
	IsBlackListed    bool
	Popularity       float64
	PriceChecks      []PriceCheck
}

type OfferComparison struct {
	HasDirectOffers   bool
	HasClickOutOffers bool
	HasBookingOffers  bool
	HasOstrovokOffers bool
	BestDirectPrice   int
	BestClickOutPrice int
	BestBookingPrice  int
	BestOstrovokPrice int
	BestTotalPrice    int
	BestOfferProvider string
}

type PriceCheck struct {
	Checkin            string
	Checkout           string
	Nights             int
	UseSearcher        bool
	DateName           string
	HasRawAPIResponses bool
	CacheError         tproto.EErrorCode
	CacheStatus        CacheStatus
	InitialStatus      CacheStatus
	Best               OfferComparison
	Breakfast          OfferComparison
	Refundable         OfferComparison
	SkipReasons        []string
	Warnings           []hproto.ESearchWarningCode
	DebugID            string
}

type HotelChecker struct {
	cache HotelCache
}

type acceptanceCheck func(HotelID string) (bool, error)
type availabilityCheck func(HotelID string, check *PriceCheck) error

type boolWithError struct {
	result bool
	err    error
}

type OffercachePermalinkInfo struct {
	IsWhitelisted bool
	IsBlackListed bool
}

type OffercachePermalinkInfoWithError struct {
	permalinkInfo OffercachePermalinkInfo
	err           error
}

type partnerSetting struct {
	Name              Partner
	AltaySlug         string
	PartnerID         string
	OperatorID        int
	AvailabilityCheck availabilityCheck
	AcceptanceCheck   acceptanceCheck
}

var partnerSettings = []partnerSetting{
	{
		Name:              Travelline,
		AltaySlug:         "ytravel_travelline",
		PartnerID:         "PI_TRAVELLINE",
		OperatorID:        44,
		AcceptanceCheck:   checkTravellineAcceptance,
		AvailabilityCheck: checkTravellineAvailability,
	},
	{
		Name:              BNovo,
		AltaySlug:         "ytravel_bnovo",
		PartnerID:         "PI_BNOVO",
		OperatorID:        45,
		AcceptanceCheck:   checkBnovoAcceptance,
		AvailabilityCheck: checkBnovoAvailability,
	},
}

var partnersByPartnerID map[string]Partner

var settingsByName map[Partner]*partnerSetting

func getPartnerSettingByName(partner Partner) *partnerSetting {
	if settingsByName == nil {
		settingsByName = make(map[Partner]*partnerSetting)
		for i := range partnerSettings {
			p := &partnerSettings[i]
			settingsByName[p.Name] = p
		}
	}
	res := settingsByName[partner]
	return res
}

type checkError struct {
	stage string
	err   error
}

func (oc *OfferComparison) Display() string {
	if !oc.HasDirectOffers {
		return "😕 " + oc.BestOfferProvider
	} else {
		if oc.BestTotalPrice == oc.BestDirectPrice {
			return "😀 " + oc.BestOfferProvider
		} else {
			return "🤨 " + oc.BestOfferProvider
		}
	}
}

func CheckHotel(ctx context.Context, cache HotelCache, partner Partner, hotelID string, checkin time.Time, checkout time.Time, useSearcher bool) (*HotelInfo, error) {
	log.Printf("Проверяем отель %s партнера %s", hotelID, partner)
	partnerCfg := getPartnerSettingByName(partner)
	if partnerCfg == nil {
		return nil, errors.New("partner not found")
	}
	info := &HotelInfo{ID: hotelID, PartnerName: partner, PartnerID: partnerCfg.PartnerID, Permalink: "неизвестно"}
	if hotelID == "" {
		return nil, errors.New("hotelId unset")
	}

	checkContext, cancellation := context.WithCancel(ctx)
	defer func() {
		cancellation()
		log.Printf("Проверка отеля %s партнера %s завершена", hotelID, partner)
	}()

	priceChecks := buildPriceChecks(checkin, checkout, useSearcher)
	geoResultChannel, permalinkChan := checkHotelStatusInGeo(checkContext, partnerCfg, hotelID)
	permalinkChannels := fanoutStringChannels(checkContext, permalinkChan, len(priceChecks)+1)
	offercachePermalinkInfoResultChannel := checkOffercachePermalinkInfoStatus(checkContext, permalinkChannels[0], partnerCfg, hotelID)

	publishingResultChannel := checkPublishingStatus(checkContext, partnerCfg, hotelID)
	acceptanceResultChannel := checkAcceptanceForPartner(checkContext, partnerCfg, hotelID)

	hotel := cache.Get(hotelID, partner)
	if hotel != nil {
		info.NameByPartner = hotel.Name
		info.IsAccessible = true
	}

	availabilityChannels := make([]<-chan error, len(priceChecks))
	offerChannels := make([]<-chan error, len(priceChecks))
	for i := range priceChecks {
		check := &priceChecks[i]
		availabilityChannels[i] = checkAvailabilityForPartner(checkContext, partnerCfg, hotelID, check)
		offerChannels[i] = checkOfferAvailability(checkContext, permalinkChannels[1+i], hotelID, partnerCfg, check)
	}

	// all queries started, now let's wait for all results:

	geoResult := <-geoResultChannel
	if geoResult.err != nil {
		return nil, checkError{stage: "проверка наличия в Гео", err: geoResult.err}
	} else {
		if geoResult.Permalink != "" {
			info.HasGeo = true
			info.Permalink = geoResult.Permalink
			info.Name = geoResult.Name
			info.URL = geoResult.URL
		} else {
			info.HasGeo = false
			info.Permalink = "неизвестно"
			info.Name = "???"
		}
	}

	offercachePermalinkInfoResult := <-offercachePermalinkInfoResultChannel
	if offercachePermalinkInfoResult.err != nil {
		if offercachePermalinkInfoResult.err.Error() != "no permalink" {
			return nil, checkError{stage: "проверка белого списка", err: offercachePermalinkInfoResult.err}
		}
	} else {
		info.IsWhiteListed = offercachePermalinkInfoResult.permalinkInfo.IsWhitelisted
		info.IsBlackListed = offercachePermalinkInfoResult.permalinkInfo.IsBlackListed
	}

	publishingResult := <-publishingResultChannel
	if publishingResult.err != nil {
		return nil, checkError{stage: "проверка публикации", err: publishingResult.err}
	} else {
		info.IsPublished = publishingResult.result
	}

	acceptanceResult := <-acceptanceResultChannel
	if acceptanceResult.err != nil {
		return nil, checkError{stage: "проверка принятия оферты", err: acceptanceResult.err}
	} else {
		info.HasAcceptedOffer = acceptanceResult.result
	}

	for i := range priceChecks {
		avCheckError := <-availabilityChannels[i]
		if avCheckError != nil {
			if avCheckError.Error() != "no permalink" {
				return nil, checkError{fmt.Sprintf("проверка офферов у партнера на дату '%s'", priceChecks[i].DateName), avCheckError}
			}
		}
		offerCheckError := <-offerChannels[i]
		if offerCheckError != nil {
			if offerCheckError.Error() != "no permalink" {
				return nil, checkError{fmt.Sprintf("проверка офферов у нас на дату '%s'", priceChecks[i].DateName), offerCheckError}
			}
		}
	}
	info.PriceChecks = priceChecks

	return info, nil
}

func (e checkError) Error() string {
	return fmt.Sprintf("Ошибка во время операции '%s': %s", e.stage, e.err)
}

func (e checkError) Unwrap() error { return e.err }

func (h *HotelInfo) HasMissingDirectOffers() bool {
	for _, check := range h.PriceChecks {
		if check.Best.HasClickOutOffers && !check.Best.HasDirectOffers {
			return true
		}
	}
	return false
}

func (h *HotelInfo) NoDirectOffersAtAll() bool {
	for _, check := range h.PriceChecks {
		if check.Best.HasDirectOffers {
			return false
		}
	}
	return true
}

func (h *HotelInfo) HasErrors() bool {
	for _, check := range h.PriceChecks {
		if check.CacheStatus != Found && check.CacheStatus != Miss {
			return true
		}
	}
	return false
}

func (h *HotelInfo) HasBadPrices() bool {
	for _, check := range h.PriceChecks {
		if check.Best.HasDirectOffers && check.Best.HasClickOutOffers && check.Best.BestClickOutPrice < check.Best.BestDirectPrice {
			return true
		}
	}
	return false
}

func (c PriceCheck) HasProblems() bool {
	return len(c.Warnings)+len(c.SkipReasons) > 0
}

func (c PriceCheck) DisplayProblems() string {
	var problems []string
	problemsMap := map[string]bool{}
	for _, warning := range c.Warnings {
		problemsMap[displayWarning(warning)] = true
	}
	for _, reason := range c.SkipReasons {
		problemsMap[displaySkipReason(reason)] = true
	}
	for problem := range problemsMap {
		problems = append(problems, problem)
	}
	return strings.Join(problems, ", ")
}

func (c PriceCheck) DisplayStatus() string {
	var status string
	var comment = ""
	switch c.CacheStatus {
	case Error:
		status = "⁉️ "
		switch c.CacheError {
		case tproto.EErrorCode_EC_GENERAL_ERROR:
			comment = "Ошибка"
		case tproto.EErrorCode_EC_NO_HOTEL_CACHE:
			comment = "Партнер собирает кеш"
		case tproto.EErrorCode_EC_DISABLED_HOTEL, tproto.EErrorCode_EC_NOT_FOUND:
			comment = "Отель недоступен"
		case tproto.EErrorCode_EC_RESOURCE_EXHAUSTED:
			comment = "Слишком много запросов"
		default:
			comment = c.CacheError.String()
		}
	case Restricted:
		status = "🛑"
		comment = "Невалидный запрос"
	case Miss:
		status = "❓"
		comment = "Нет данных"
	case Empty:
		status = "❌"
	case Found:
		status = "✅"
	}
	return status + " " + comment
}

func (c PriceCheck) HasResponse() bool {
	return c.CacheStatus == Empty || c.CacheStatus == Found
}

func checkAcceptanceForPartner(ctx context.Context, partner *partnerSetting, hotelID string) <-chan boolWithError {
	resultChannel := make(chan boolWithError)
	go func() {
		defer close(resultChannel)
		res, err := partner.AcceptanceCheck(hotelID)
		select {
		case resultChannel <- boolWithError{res, err}:
		case <-ctx.Done():
			return
		}
	}()
	return resultChannel
}

func checkAvailabilityForPartner(ctx context.Context, partner *partnerSetting, hotelID string, check *PriceCheck) <-chan error {
	resultChannel := make(chan error)
	go func() {
		defer close(resultChannel)
		err := partner.AvailabilityCheck(hotelID, check)
		select {
		case resultChannel <- err:
		case <-ctx.Done():
			return
		}
	}()
	return resultChannel
}

func fanoutStringChannels(ctx context.Context, input <-chan string, num int) []chan string {
	channels := make([]chan string, num)
	for i := range channels {
		channels[i] = make(chan string, 1)
	}
	go func() {
		defer func() {
			for i := 0; i < num; i++ {
				close(channels[i])
			}
		}()
		inp := <-input
		if inp == "" {
			return
		}
		for i := 0; i < num; i++ {
			select {
			case channels[i] <- inp:
			case <-ctx.Done():
				return
			}
		}
	}()
	return channels
}

func GetPartnerByID(ID string) Partner {
	if partnersByPartnerID == nil {
		partnersByPartnerID = make(map[string]Partner)
		for i := range partnerSettings {
			p := &partnerSettings[i]
			partnersByPartnerID[p.PartnerID] = p.Name
		}
	}
	res := partnersByPartnerID[ID]
	return res
}

func buildPriceChecks(checkin time.Time, checkout time.Time, useSearcher bool) []PriceCheck {
	if checkin.IsZero() {
		tomorrow := time.Now().Add(time.Hour * 24)
		var onWeekend time.Time
		for date := tomorrow; true; date = date.Add(time.Hour * 24) {
			if date.Weekday() == time.Saturday {
				onWeekend = date
				break
			}
		}
		inOneMonth := time.Now().AddDate(0, 1, 0)
		inTwoWeeks := time.Now().AddDate(0, 0, 14)
		inTwoMonths := time.Now().AddDate(0, 2, 0)
		res := []PriceCheck{
			{
				Checkin:     tomorrow.Format(dateLayout),
				Checkout:    tomorrow.AddDate(0, 0, 1).Format(dateLayout),
				Nights:      1,
				DateName:    "Завтра на одну ночь",
				UseSearcher: useSearcher,
			},
			{
				Checkin:     onWeekend.Format(dateLayout),
				Checkout:    onWeekend.AddDate(0, 0, 1).Format(dateLayout),
				Nights:      1,
				DateName:    "В выходные на одну ночь",
				UseSearcher: useSearcher,
			},
			{
				Checkin:     inTwoWeeks.Format(dateLayout),
				Checkout:    inTwoWeeks.AddDate(0, 0, 5).Format(dateLayout),
				Nights:      5,
				DateName:    "Через две недели на пять ночей",
				UseSearcher: useSearcher,
			},
			{
				Checkin:     inOneMonth.Format(dateLayout),
				Checkout:    inOneMonth.AddDate(0, 0, 1).Format(dateLayout),
				Nights:      1,
				DateName:    "Через месяц на одну ночь",
				UseSearcher: useSearcher,
			},
			{
				Checkin:     inTwoMonths.Format(dateLayout),
				Checkout:    inTwoMonths.AddDate(0, 0, 3).Format(dateLayout),
				Nights:      3,
				DateName:    "Через два месяца на три ночи",
				UseSearcher: useSearcher,
			},
		}
		for i := range res {
			pc := &res[i]
			pc.DateName = fmt.Sprintf("%s (%s — %s)", pc.DateName, pc.Checkin, pc.Checkout)
		}
		return res
	} else {
		if checkout.IsZero() {
			checkout = checkin.AddDate(0, 0, 1)
		}
		dur := checkout.Sub(checkin)
		nights := int(math.Ceil(dur.Hours() / 24.0))
		return []PriceCheck{
			{
				Checkin:     checkin.Format(dateLayout),
				Checkout:    checkout.Format(dateLayout),
				Nights:      nights,
				DateName:    fmt.Sprintf("%s - %s", checkin.Format(dateLayout), checkout.Format(dateLayout)),
				UseSearcher: useSearcher,
			},
		}
	}
}
