package core

import (
	"context"
	"encoding/gob"
	"fmt"
	"log"
	"os"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/travel/hotels/tools/boy_hotels_checker/internal/cache"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

type HotelCacheItem struct {
	Name    string
	ID      string
	Partner Partner
}

type HotelCacheConfiguration struct {
	CacheFileName string
	CacheYtPath   string
	YtToken       string
	YtProxy       string
	SaveChanges   bool
}

type HotelCache interface {
	Get(hotelID string, partner Partner) *HotelCacheItem
	Remove(hotelID string, partner Partner)
	Put(item *HotelCacheItem)
	PutAll(items []*HotelCacheItem)
	Lookup(query string, limit int) []*HotelCacheItem
	All() []*HotelCacheItem
}

type HotelCacheUpdateEventType uint

const (
	NewHotel HotelCacheUpdateEventType = iota
	RemovedHotel
)

type HotelCacheUpdateEvent struct {
	EventType HotelCacheUpdateEventType
	Hotel     *HotelCacheItem
	Err       error
}

type HotelUpdater func(hotelCache HotelCache) ([]HotelCacheUpdateEvent, error)

type HotelCacheUpdateEventHandler func(event HotelCacheUpdateEvent)

type UpdatableHotelCache struct {
	cache.Cache
	config   HotelCacheConfiguration
	ytClient yt.Client
	handlers []HotelCacheUpdateEventHandler
}

type compositeID struct {
	partner Partner
	hotelID string
}

func (h *UpdatableHotelCache) Get(hotelID string, partner Partner) *HotelCacheItem {
	id := compositeID{
		partner: partner,
		hotelID: hotelID,
	}
	res, found := h.Cache.Get(id)
	if !found {
		return nil
	}
	hci := res.(*HotelCacheItem)
	return hci
}

func (h *UpdatableHotelCache) Remove(hotelID string, partner Partner) {
	h.Cache.Remove(compositeID{
		partner: partner,
		hotelID: hotelID,
	})
}

func (h *UpdatableHotelCache) Put(item *HotelCacheItem) {
	id := compositeID{
		partner: item.Partner,
		hotelID: item.ID,
	}
	h.Cache.Put(id, item)
}

func (h *UpdatableHotelCache) PutAll(items []*HotelCacheItem) {
	var keys = make([]interface{}, len(items))
	var values = make([]interface{}, len(items))
	for i := range items {
		keys[i] = compositeID{
			partner: items[i].Partner,
			hotelID: items[i].ID,
		}
		values[i] = items[i]
	}
	h.Cache.PutAll(keys, values)
}

func (h *UpdatableHotelCache) Lookup(query string, limit int) []*HotelCacheItem {
	queried := h.Cache.Lookup(query, limit)
	res := make([]*HotelCacheItem, len(queried))
	for i := range queried {
		res[i] = queried[i].(*HotelCacheItem)
	}
	return res
}

func (h *UpdatableHotelCache) All() []*HotelCacheItem {
	all := h.Cache.All()
	res := make([]*HotelCacheItem, len(all))
	for i := range all {
		res[i] = all[i].(*HotelCacheItem)
	}
	return res
}

func (h *UpdatableHotelCache) Subscribe(handler HotelCacheUpdateEventHandler) {
	h.handlers = append(h.handlers, handler)
}

func (h *UpdatableHotelCache) ScheduleUpdates(updateInterval time.Duration, ctx context.Context) {
	ticker := time.NewTicker(updateInterval)
	go func() {
		for {
			select {
			case <-ticker.C:
				h.Update(ctx)
			case <-ctx.Done():
				ticker.Stop()
				return
			}
		}
	}()
}

func (h *UpdatableHotelCache) saveToYt(ctx context.Context) error {
	wctx, cancel := context.WithCancel(ctx)
	defer cancel()
	tx, err := h.ytClient.BeginTx(wctx, nil)
	if err != nil {
		return fmt.Errorf("unable to save hotels cache: %w", err)
	}

	path := ypath.Path(h.config.CacheYtPath)
	_, err = tx.CreateNode(wctx, path, yt.NodeTable, &yt.CreateNodeOptions{
		Force:     true,
		Recursive: true,
	})
	if err != nil {
		return fmt.Errorf("unable to save hotels cache: %w", err)
	}
	wrr, err := tx.WriteTable(wctx, path, nil)
	if err != nil {
		return fmt.Errorf("unable to save hotels cache: %w", err)
	}

	for _, item := range h.All() {
		err = wrr.Write(item)
		if err != nil {
			return fmt.Errorf("unable to save hotels cache: %w", err)
		}
	}

	err = wrr.Commit()
	if err != nil {
		return fmt.Errorf("unable to save hotels cache: %w", err)
	}
	err = tx.Commit()
	if err != nil {
		return fmt.Errorf("unable to save hotels cache: %w", err)
	}
	return nil
}

func (h *UpdatableHotelCache) save(ctx context.Context) error {
	if h.ytClient != nil && h.config.CacheYtPath != "" {
		ytErr := h.saveToYt(ctx)
		if ytErr != nil {
			log.Println(ytErr)
		}
	}

	if h.config.CacheFileName != "" {
		f, err := os.Create(h.config.CacheFileName)
		if err != nil {
			log.Printf("Unable to open %s for writing: %s", h.config.CacheFileName, err)
			return err
		}
		defer f.Close()
		encoder := gob.NewEncoder(f)
		if err := encoder.Encode(h.All()); err != nil {
			log.Printf("Unable to save hotel data: %s", err)
			return err
		}
	}
	return nil
}

func (h *UpdatableHotelCache) load(ctx context.Context) error {
	if h.ytClient != nil && h.config.CacheYtPath != "" {
		err := h.loadFromYt(ctx)
		if err == nil {
			return nil
		} else {
			log.Println(err)
		}
	}

	log.Println("No cache in yt, falling back to file")
	var results []*HotelCacheItem
	if h.config.CacheFileName != "" {
		f, err := os.Open(h.config.CacheFileName)
		if err != nil {
			return fmt.Errorf("unable to open %s for reading: %w", h.config.CacheFileName, err)
		}
		defer f.Close()
		decoder := gob.NewDecoder(f)
		if err := decoder.Decode(&results); err != nil {
			return fmt.Errorf("unable to load hotel data: %w", err)
		}
		h.PutAll(results)
	}
	return nil
}

func (h *UpdatableHotelCache) loadFromYt(ctx context.Context) error {
	rctx, cancel := context.WithCancel(ctx)
	log.Println("Loading hotel cache from yt")
	defer func() {
		cancel()
		log.Println("Done loading yt hotel cache")
	}()
	tx, err := h.ytClient.BeginTx(rctx, nil)
	if err != nil {
		return fmt.Errorf("unable to load hotels cache: %w", err)
	}

	var items []*HotelCacheItem
	rdr, err := tx.ReadTable(rctx, ypath.Path(h.config.CacheYtPath), nil)
	if err != nil {
		return fmt.Errorf("unable to load hotels cache: %w", err)
	}

	for rdr.Next() {
		var item HotelCacheItem
		err := rdr.Scan(&item)
		if err != nil {
			return fmt.Errorf("unable to load hotels cache: %w", err)
		}
		items = append(items, &item)
	}
	if err := rdr.Err(); err != nil {
		return err
	}

	h.PutAll(items)
	return nil
}

func (h *UpdatableHotelCache) Update(ctx context.Context) {
	log.Println("Will update caches")
	defer log.Println("Done updating caches")
	wg := sync.WaitGroup{}
	wg.Add(len(partnerHotelUpdaters))

	numberOfUpdates := map[Partner]int{}
	for p, u := range partnerHotelUpdaters {
		update := u
		partner := p
		done := make(chan struct{})
		go func() {
			defer close(done)
			events, err := update(h)
			if err != nil {
				log.Println(fmt.Errorf("error while updating cache for partner %s: %w", partner, err))
			} else {
				numberOfUpdates[partner] = len(events)
				for _, e := range events {
					for _, h := range h.handlers {
						go h(e)
					}
				}
			}
		}()
		go func() {
			select {
			case <-done:
				log.Printf("Partner %s cache update completed normally with %d updates", partner, numberOfUpdates[partner])
				wg.Done()
			case <-ctx.Done():
				log.Printf("Partner %s cache update interrupted as context is closed", partner)
				wg.Done()
			}
		}()
	}
	wg.Wait()
	for _, num := range numberOfUpdates {
		if num > 0 {
			log.Printf("Caches updated, will save")
			_ = h.save(ctx)
			return
		}
	}
	log.Printf("No changes found")
}

func queryMatcher(item interface{}, query string) bool {
	if query == "" {
		return true
	}
	hci := item.(*HotelCacheItem)
	return strings.Contains(strings.ToLower(hci.Name), query)
}

func NewHotelCache(ctx context.Context, configuration HotelCacheConfiguration) UpdatableHotelCache {
	var ytClient yt.Client
	if configuration.CacheYtPath != "" {
		ytConfig := &yt.Config{
			Proxy: configuration.YtProxy,
			Token: configuration.YtToken,
		}
		var err error
		ytClient, err = ythttp.NewClient(ytConfig)
		if err != nil {
			log.Println(fmt.Errorf("unable to initialize yt client for hotel cache: %w", err))
		}
	}

	result := UpdatableHotelCache{
		Cache:    cache.NewCache(ctx, queryMatcher),
		config:   configuration,
		ytClient: ytClient,
	}
	if err := result.load(ctx); err != nil {
		log.Println(fmt.Errorf("unable to load cache: %w", err))
	}

	return result
}
