package core

import (
	hproto "a.yandex-team.ru/travel/hotels/proto2"
	"a.yandex-team.ru/travel/hotels/tools/boy_hotels_checker/pkg/utils"
	tproto "a.yandex-team.ru/travel/proto"
	"context"
	"errors"
	"fmt"
	"github.com/gofrs/uuid"
	"github.com/valyala/fastjson"
	"net/http"
	"net/url"
	"strconv"
	"time"
)

func checkWhitelistStatusImpl(hotelID string, permalink string, partner *partnerSetting) (OffercachePermalinkInfo, error) {
	args := url.Values{}
	args.Add("Permalink", permalink)
	parsed, err := utils.HTTPGet(offerCacheURL.Value+"/permalink", args, http.Header{})
	var result OffercachePermalinkInfo
	if err != nil {
		return result, err
	}
	for _, hotel := range parsed.GetArray("PartnerHotels") {
		name := string(hotel.GetStringBytes("PartnerId"))
		hid := string(hotel.GetStringBytes("OriginalId"))
		if name == partner.PartnerID && hid == hotelID {
			result.IsWhitelisted = hotel.GetBool("IsWhitelisted")
			result.IsBlackListed = hotel.GetBool("IsBlacklisted")
			return result, nil
		}
	}
	return result, nil
}

func checkOffercachePermalinkInfoStatus(ctx context.Context, permalinkChannel <-chan string, partner *partnerSetting, hotelID string) chan OffercachePermalinkInfoWithError {
	resultChannel := make(chan OffercachePermalinkInfoWithError)
	go func() {
		defer close(resultChannel)
		var result OffercachePermalinkInfoWithError
		permalink := <-permalinkChannel
		if permalink != "" {
			result.permalinkInfo, result.err = checkWhitelistStatusImpl(hotelID, permalink, partner)
		} else {
			result.err = errors.New("no permalink")
		}
		select {
		case resultChannel <- result:
		case <-ctx.Done():
			return
		}
	}()
	return resultChannel
}

func findOffers(permalink string, date string, nights int, request int, useSearcher bool, debugID string) (*fastjson.Value, error) {
	args := url.Values{}
	args.Add("SHotelId", fmt.Sprintf("%s~f.single_org", permalink))
	args.Add("Date", date)
	args.Add("Debug", "1")
	args.Add("Nights", strconv.Itoa(nights))
	args.Add("Ages", "88,88")
	args.Add("GeoOrigin", "hotels-checker-tg-bot")
	args.Add("YaTravelDebugId", debugID)
	args.Add("RequestClass", "RC_BACKGROUND")
	if useSearcher {
		args.Add("UseSearcher", "1")
	} else {
		args.Add("UseSearcher", "0")
	}
	args.Add("Full", "1")
	args.Add("RequestId", strconv.Itoa(request))
	return utils.HTTPGet(offerCacheURL.Value+"/read", args, http.Header{})
}

func checkOfferAvailabilityImpl(permalink string, hotelID string, partner *partnerSetting, check *PriceCheck) error {
	uid, _ := uuid.DefaultGenerator.NewV4()
	check.DebugID = uid.String()
	request := 0
	var result *fastjson.Value
	for {
		if request >= 100 {
			break
		}
		var err error
		result, err = findOffers(permalink, check.Checkin, check.Nights, request, check.UseSearcher, uid.String())
		if err != nil {
			return fmt.Errorf("unable to complete offer search: %w", err)
		}
		if !result.Exists("Debug", "HotelExtras", permalink, "SubHotels") {
			return errors.New("no debug info in response")
		}
		for _, sh := range result.GetArray("Debug", "HotelExtras", permalink, "SubHotels") {
			partnerID := string(sh.GetStringBytes("HotelId", "PartnerId"))
			originalID := string(sh.GetStringBytes("HotelId", "OriginalId"))
			if partnerID == partner.PartnerID && originalID == hotelID {
				status := CacheStatus(sh.GetStringBytes("CacheStatus"))
				if request == 0 {
					check.InitialStatus = status
				}
				check.CacheStatus = status
				if sh.Exists("CacheErrorCode") {
					cacheError := string(sh.GetStringBytes("CacheErrorCode"))
					code, err := errorCodeFromString(cacheError)
					if err == nil {
						check.CacheError = code
					}
				}
				if sh.Exists("CacheRecords") {
					for _, cr := range sh.GetArray("CacheRecords") {
						if cr.Exists("SearchWarnings", "Warnings") {
							for _, sw := range cr.GetArray("SearchWarnings", "Warnings") {
								warning := string(sw.GetStringBytes("Code"))
								warningCode, err := warningFromString(warning)
								if err == nil {
									check.Warnings = append(check.Warnings, warningCode)
								}
							}
						}
					}
				}
				break
			}
		}
		if !check.UseSearcher || result.GetBool("IsFinished") {
			break
		}
		request++
		result = nil
		time.Sleep(time.Second)
	}
	if result == nil {
		return errors.New("unable to complete search in 100 attempts")
	}
	if result.Exists("Debug", "HotelExtras", permalink, "SkippedPrices") {
		reasons := map[string]bool{}
		for _, sp := range result.GetArray("Debug", "HotelExtras", permalink, "SkippedPrices") {
			opID := sp.GetInt("OperatorId")
			if opID == partner.OperatorID {
				reason := string(sp.GetStringBytes("SkipReason"))
				reasons[reason] = true
			}
		}
		for reason := range reasons {
			check.SkipReasons = append(check.SkipReasons, reason)
		}
	}
	if result.Exists("Hotels", permalink, "Prices") {
		check.Best = OfferComparison{}
		check.Breakfast = OfferComparison{}
		check.Refundable = OfferComparison{}

		prices := result.GetArray("Hotels", permalink, "Prices")
		for i := range prices {
			priceObj := prices[i]
			price := priceObj.GetInt("Price")
			opID := priceObj.GetInt("OperatorId")
			freeCancellation := priceObj.GetBool("FreeCancellation")
			hasBreakfast := priceObj.GetBool("BreakfastIncluded")
			if opID == partner.OperatorID {
				check.Best.HasDirectOffers = true
				check.Best.BestDirectPrice = utils.MinNotZero(check.Best.BestDirectPrice, price)
				if freeCancellation {
					check.Refundable.HasDirectOffers = true
					check.Refundable.BestDirectPrice = utils.MinNotZero(check.Refundable.BestDirectPrice, price)
				}
				if hasBreakfast {
					check.Breakfast.HasDirectOffers = true
					check.Breakfast.BestDirectPrice = utils.MinNotZero(check.Breakfast.BestDirectPrice, price)
				}
			} else {
				hasBookingOffers := false
				hasOstrovokOffers := false
				bestBookingPrice := 0
				bestOstrovokPrice := 0
				switch opID {
				case 2:
					hasBookingOffers = true
					bestBookingPrice = utils.MinNotZero(bestBookingPrice, price)
				case 4:
					hasOstrovokOffers = true
					bestOstrovokPrice = utils.MinNotZero(bestOstrovokPrice, price)
				}
				check.Best.HasClickOutOffers = true
				check.Best.BestClickOutPrice = utils.MinNotZero(check.Best.BestClickOutPrice, price)
				check.Best.HasBookingOffers = hasBookingOffers
				check.Best.HasOstrovokOffers = hasOstrovokOffers
				check.Best.BestBookingPrice = bestBookingPrice
				check.Best.BestOstrovokPrice = bestOstrovokPrice
				if hasBreakfast {
					check.Breakfast.HasClickOutOffers = true
					check.Breakfast.BestClickOutPrice = utils.MinNotZero(check.Breakfast.BestClickOutPrice, price)
					check.Breakfast.HasBookingOffers = hasBookingOffers
					check.Breakfast.HasOstrovokOffers = hasOstrovokOffers
					check.Breakfast.BestBookingPrice = bestBookingPrice
					check.Breakfast.BestOstrovokPrice = bestOstrovokPrice
				}
				if freeCancellation {
					check.Refundable.HasClickOutOffers = true
					check.Refundable.BestClickOutPrice = utils.MinNotZero(check.Refundable.BestClickOutPrice, price)
					check.Refundable.HasBookingOffers = hasBookingOffers
					check.Refundable.HasOstrovokOffers = hasOstrovokOffers
					check.Refundable.BestBookingPrice = bestBookingPrice
					check.Refundable.BestOstrovokPrice = bestOstrovokPrice
				}

			}
			if price < check.Best.BestTotalPrice || (price == check.Best.BestTotalPrice && opID == partner.OperatorID) || check.Best.BestTotalPrice == 0 {
				check.Best.BestTotalPrice = price
				check.Best.BestOfferProvider = enhanceOpName(string(priceObj.GetStringBytes("OperatorName")), opID)
			}
			if hasBreakfast {
				if price < check.Breakfast.BestTotalPrice || (price == check.Breakfast.BestTotalPrice && opID == partner.OperatorID) || check.Breakfast.BestTotalPrice == 0 {
					check.Breakfast.BestTotalPrice = price
					check.Breakfast.BestOfferProvider = enhanceOpName(string(priceObj.GetStringBytes("OperatorName")), opID)
				}
			}
			if freeCancellation {
				if price < check.Refundable.BestTotalPrice || (price == check.Refundable.BestTotalPrice && opID == partner.OperatorID) || check.Refundable.BestTotalPrice == 0 {
					check.Refundable.BestTotalPrice = price
					check.Refundable.BestOfferProvider = enhanceOpName(string(priceObj.GetStringBytes("OperatorName")), opID)
				}
			}
		}
	}
	return nil
}

func enhanceOpName(opName string, opID int) string {
	if opName == "Яндекс.Путешествия" {
		switch opID {
		case 40:
			return "Экспедия"
		case 43:
			return "Дельфин"
		case 44:
			return "Travelline"
		case 45:
			return "Bnovo"
		}
	}
	return opName
}

func errorCodeFromString(value string) (tproto.EErrorCode, error) {
	i, found := tproto.EErrorCode_value[value]
	if found {
		return tproto.EErrorCode(i), nil
	} else {
		return 0, errors.New("unsupported Error Code")
	}
}

func warningFromString(value string) (hproto.ESearchWarningCode, error) {
	i, found := hproto.ESearchWarningCode_value[value]
	if found {
		return hproto.ESearchWarningCode(i), nil
	} else {
		return hproto.ESearchWarningCode_SW_UNUSED, errors.New("unsupported warning code")
	}
}

func displaySkipReason(reason string) string {
	switch reason {
	case "SR_OperatorDisabled":
		return "Партнер отключен"
	case "SR_NonDistinguishable":
		return "Неразличимые тарифы"
	case "SR_BannedTravellineRatePlan", "SR_BannedBNovoRatePlan":
		return "Черный список ТП"
	default:
		return reason
	}
}

func displayWarning(code hproto.ESearchWarningCode) string {
	switch code {
	case hproto.ESearchWarningCode_SW_TL_DROPPED_APARTMENTS:
		return "Апартаменты по другому адресу"
	case hproto.ESearchWarningCode_SW_TL_DROPPED_RATEPLAN_WITH_TREATMENT:
		return "Тариф с лечением"
	case hproto.ESearchWarningCode_SW_TL_DROPPED_UNKNOWN_RATE_PLAN, hproto.ESearchWarningCode_SW_TL_DROPPED_NON_YANDEX_RATE_PLAN, hproto.ESearchWarningCode_SW_BN_DROPPED_DISABLED_RATE_PLAN:
		return "Недоступный тариф"
	case hproto.ESearchWarningCode_SW_BN_DROPPED_FORBIDDEN_OFFERS:
		return "Цена -1 (обычно значит, что тарифы выключены в bnovo)"
	case hproto.ESearchWarningCode_SW_TL_DROPPED_MISSING_HOTEL_DATA:
		return "Нехватка данных"
	case hproto.ESearchWarningCode_SW_TL_DROPPED_INVALID_ALLOCATION, hproto.ESearchWarningCode_SW_TL_DROPPED_NO_PLACEMENTS:
		return "Гости не влезают"
	case hproto.ESearchWarningCode_SW_TL_DROPPED_INVALID_CANCELLATION_PENALTIES, hproto.ESearchWarningCode_SW_BN_DROPPED_INVALID_CANCELLATION_PENALTIES:
		return "Неверные правила отмены"
	case hproto.ESearchWarningCode_SW_TL_DROPPED_REQUIRES_CHILDREN:
		return "Тариф только с детьми"
	case hproto.ESearchWarningCode_SW_BN_DROPPED_HOSTEL_BEDS:
		return "Неподдерживаемые койкоместа"
	default:
		return code.String()
	}
}

func checkOfferAvailability(ctx context.Context, permalinkChannel <-chan string, hotelID string, partner *partnerSetting, check *PriceCheck) <-chan error {
	resultChannel := make(chan error)
	go func() {
		defer close(resultChannel)
		permalink := <-permalinkChannel
		var err error
		if permalink == "" {
			err = errors.New("no permalink")
		} else {
			err = checkOfferAvailabilityImpl(permalink, hotelID, partner, check)
		}
		select {
		case resultChannel <- err:
		case <-ctx.Done():
			return
		}
	}()

	return resultChannel
}
