package core

import (
	"a.yandex-team.ru/travel/hotels/tools/boy_hotels_checker/pkg/utils"
	"fmt"
	"log"
	"net/http"
	"net/url"
)

func travellineHotelUpdater(cache HotelCache) ([]HotelCacheUpdateEvent, error) {
	var result []HotelCacheUpdateEvent
	existingCopy := make(map[string]*HotelCacheItem)
	for _, v := range cache.All() {
		if v.Partner == Travelline {
			existingCopy[v.ID] = v
		}
	}
	args := url.Values{}
	headers := http.Header{}
	headers.Add("X-ApiKey", tlAvailabilityAPIKey.Value)
	res, err := utils.HTTPGet(travellineAPI.Value+"/booking/hotels", args, headers)
	if err != nil {
		return nil, fmt.Errorf("unable to update traveline cache: %w", err)
	}
	if !res.Exists("hotels") {
		return nil, fmt.Errorf("unable to update travelline cache: no hotels in response")
	}
	hotels := res.GetArray("hotels")
	for i := range hotels {
		if inventoryVersion := hotels[i].GetInt64("inventory_version"); inventoryVersion == 0 {
			continue
		}
		hotelID := string(hotels[i].GetStringBytes("code"))
		delete(existingCopy, hotelID)
		found := cache.Get(hotelID, Travelline)
		if found == nil {
			name, err := getTLHotelName(hotelID)
			r := HotelCacheUpdateEvent{
				EventType: NewHotel,
			}
			if err != nil {
				r.Err = err
			} else {
				log.Printf("New Travelline hotel with id '%s' and name '%s'", hotelID, name)
				item := &HotelCacheItem{
					Name:    name,
					ID:      hotelID,
					Partner: Travelline,
				}
				cache.Put(item)
				r.Hotel = item
			}
			result = append(result, r)
		}
	}
	for id, item := range existingCopy {
		log.Printf("Removing Travelline hotel with id '%s' and name '%s'", id, item.Name)
		cache.Remove(id, Travelline)
		r := HotelCacheUpdateEvent{
			EventType: RemovedHotel,
			Hotel:     item,
		}
		result = append(result, r)
	}
	return result, nil
}

func getTLHotelName(hotelID string) (string, error) {
	q := url.Values{}
	q.Add("hotel.code", hotelID)
	q.Add("language", "ru-ru")
	h := http.Header{}
	h.Add("x-apikey", tlAvailabilityAPIKey.Value)
	parsed, err := utils.HTTPGet(travellineAPI.Value+"/booking/hotel_info", q, h)
	if err != nil {
		return "", err
	}
	if parsed.Exists("hotel", "name") {
		return string(parsed.GetStringBytes("hotel", "name")), nil
	}
	return "", nil
}

func checkTravellineAvailability(hotelID string, check *PriceCheck) error {
	args := url.Values{}
	args.Add("hotel.code", hotelID)
	args.Add("start_date", check.Checkin)
	args.Add("end_date", check.Checkout)
	h := http.Header{}
	h.Add("x-apikey", tlAvailabilityAPIKey.Value)
	parsed, err := utils.HTTPGet(travellineAPI.Value+"/booking/hotel_offer_availability", args, h)
	if err != nil {
		return err
	}
	if parsed.Exists("room_stays") {
		numStays := len(parsed.GetArray("room_stays"))
		check.HasRawAPIResponses = numStays > 0
	}
	return nil
}

func checkTravellineAcceptance(hotelID string) (bool, error) {
	args := url.Values{}
	args.Add("hotel.code", hotelID)
	h := http.Header{}
	h.Add("x-apikey", tlConnectionAPIKey.Value)
	parsed, err := utils.HTTPGet(travellineAPI.Value+"/offer/hotel_details", args, h)
	if err != nil {
		return false, err
	}
	if parsed.Exists("hotel_details", "offer_status") {
		offerStatus := string(parsed.GetStringBytes("hotel_details", "offer_status"))
		if offerStatus == "accepted" {
			return true, nil
		}
	}
	return false, nil
}
