package cfg

import (
	"a.yandex-team.ru/travel/library/go/vault"
	"encoding/json"
	"flag"
	"fmt"
	"log"
	"os"
	"strings"
)

const argsReplaceEnvVar = "ARG_REPLACES"

type ConfigurableSetting struct {
	Value       string
	Name        string
	Description string
	SecretID    string
	SecretName  string
	EnvVarName  string
	LoadFromEnv bool
	Required    bool
	IsSecret    bool
}

var YavToken = ConfigurableSetting{
	Value:       "",
	Name:        "vault-token",
	Description: "Oauth token to access yandex vault to access other secrets",
	LoadFromEnv: true,
	IsSecret:    true,
}

func (s *ConfigurableSetting) register() {
	description := s.Description
	if s.LoadFromEnv && s.Value == "" {
		var envName string
		if s.EnvVarName != "" {
			envName = s.EnvVarName
		} else {
			envName = strings.ToUpper(strings.ReplaceAll(s.Name, "-", "_"))
		}
		description = fmt.Sprintf("%s (default to %s env var)", description, envName)
	}
	flag.StringVar(&s.Value, s.Name, s.Value, description)
}

func (s *ConfigurableSetting) load(resolver *vault.YavSecretsResolver, replacements map[string]string) bool {
	if s.Value == "" {
		if s.LoadFromEnv {
			var envName string
			if s.EnvVarName != "" {
				envName = s.EnvVarName
			} else {
				envName = strings.ToUpper(strings.ReplaceAll(s.Name, "-", "_"))
			}
			envVal := os.Getenv(envName)
			if envVal != "" {
				s.Value = envVal
				return true
			}
		}
		if s.SecretID != "" && s.SecretName != "" && resolver != nil {
			val, err := resolver.GetSecretValue(s.SecretID, s.SecretName)
			if err != nil {
				log.Printf("Error while resolving secret %s: %s", s.Name, err.Error())
			} else if val != "" {
				s.Value = val
				return true
			}
		}
		if s.Required {
			return false
		}
	} else {
		if val, exists := replacements[s.Value]; exists {
			log.Printf("Will replace the value of var %s with a value from %s", s.Name, argsReplaceEnvVar)
			s.Value = val
			return true
		}
	}
	return true
}

func LoadSettings(settings ...*ConfigurableSetting) error {
	var yavPresent bool
	for i := range settings {
		if settings[i] == &YavToken {
			yavPresent = true
		}
		settings[i].register()
	}
	flag.Parse()
	var replacements map[string]string
	if replVal, exists := os.LookupEnv(argsReplaceEnvVar); exists {
		err := json.Unmarshal([]byte(replVal), &replacements)
		if err != nil {
			return fmt.Errorf("unable to marshal settings: %w", err)
		}
	}

	var uninitialized []string
	var resolver *vault.YavSecretsResolver
	if yavPresent {
		YavToken.load(nil, replacements)
		if YavToken.Value != "" {
			resolver = vault.NewYavSecretsResolverFromToken(YavToken.Value)
		}
	}
	for i := range settings {
		if !settings[i].load(resolver, replacements) {
			uninitialized = append(uninitialized, settings[i].Name)
		}
		value := settings[i].Value
		if settings[i].IsSecret {
			value = "***SECRET***"
		}
		log.Printf("%s = %s", settings[i].Name, value)
	}
	if len(uninitialized) > 0 {
		return fmt.Errorf("the following required configuration properties were neither specified nor resolved: %s",
			strings.Join(uninitialized, ", "))
	}

	return nil
}
