package excel

import (
	"a.yandex-team.ru/library/go/x/math"
	"errors"
	"fmt"
	"github.com/xuri/excelize/v2"
	"reflect"
	"strconv"
	"strings"
)

type excelColumn struct {
	Name                string
	Width               int
	NonZero             bool
	Centered            bool
	SubColumns          []*excelColumn
	HyperlinkFieldIndex int
	FieldIndex          int
}

type schema struct {
	columns []*excelColumn
	width   int
	height  int
}

type ReportPage struct {
	Title       string
	SheetName   string
	Description string
	Items       []interface{}
}

func (c *excelColumn) getDimensions() (int, int) {
	width := 0
	maxNestedHeight := 0
	if len(c.SubColumns) == 0 {
		width = 1
	} else {
		for _, col := range c.SubColumns {
			nestedWidth, nestedHeight := col.getDimensions()
			width += nestedWidth
			maxNestedHeight = math.MaxInt(maxNestedHeight, nestedHeight)
		}
	}
	return width, 1 + maxNestedHeight
}

func (c *excelColumn) printHeaders(f *excelize.File, sheet string, row int, column int, lastRow int) error {
	if err := f.SetCellStr(sheet, cell(row, column), c.Name); err != nil {
		return err
	}
	style, err := f.NewStyle(`{"alignment":{"horizontal":"center", "vertical":"center"}}`)
	if err != nil {
		return err
	}
	if err := f.SetCellStyle(sheet, cell(row, column), cell(row, column), style); err != nil {
		return err
	}
	if len(c.SubColumns) == 0 {
		if c.Width > 0 {
			if err := f.SetColWidth(sheet, col(column), col(column), float64(c.Width)); err != nil {
				return err
			}
		}
		if row < lastRow {
			if err := f.MergeCell(sheet, cell(row, column), cell(lastRow, column)); err != nil {
				return err
			}
		}
	} else {
		width, _ := c.getDimensions()
		if width > 1 {
			if err := f.MergeCell(sheet, cell(row, column), cell(row, column+width-1)); err != nil {
				return err
			}
		}
	}

	colNum := column
	for _, nested := range c.SubColumns {
		if err := nested.printHeaders(f, sheet, row+1, colNum, lastRow); err != nil {
			return err
		}
		width, _ := nested.getDimensions()
		colNum += width
	}
	return nil
}

func parseField(objectType reflect.Type, index int) (*excelColumn, error) {
	field := objectType.Field(index)
	tag, present := field.Tag.Lookup("excel")
	if !present {
		return nil, nil
	}
	column := excelColumn{
		FieldIndex:          index,
		HyperlinkFieldIndex: -1,
	}
	parts := strings.Split(tag, ",")
	if parts[0] != "" {
		column.Name = parts[0]
	} else {
		column.Name = field.Name
	}
	if len(parts) > 0 {
		for _, part := range parts[1:] {
			subparts := strings.Split(part, "=")
			if len(subparts) > 2 {
				continue
			}
			key := subparts[0]
			var val string
			if len(subparts) == 2 {
				val = subparts[1]
			}
			switch key {
			case "width":
				width, err := strconv.Atoi(val)
				if err == nil {
					column.Width = width
				}
			case "nonzero":
				column.NonZero = true
			case "centered":
				column.Centered = true
			case "link":
				for i := 0; i < objectType.NumField(); i++ {
					if objectType.Field(i).Name == val {
						column.HyperlinkFieldIndex = i
					}
				}
				if column.HyperlinkFieldIndex == -1 {
					return nil, errors.New("hyperlink field not found")
				}
			}
		}
	}

	fieldType := field.Type
	if fieldType.Kind() == reflect.Ptr {
		fieldType = fieldType.Elem()
	}

	if fieldType.Kind() == reflect.Struct {
		for i := 0; i < fieldType.NumField(); i++ {
			nested, err := parseField(fieldType, i)
			if err != nil {
				return nil, err
			}
			if nested != nil {
				column.SubColumns = append(column.SubColumns, nested)
			}
		}
	}
	return &column, nil
}

func buildSchema(objectType reflect.Type) (*schema, error) {
	var columns []*excelColumn
	maxHeight := 0
	maxWidth := 0
	for i := 0; i < objectType.NumField(); i++ {
		column, err := parseField(objectType, i)
		if err != nil {
			return nil, err
		}
		if column != nil {
			width, height := column.getDimensions()
			maxHeight = math.MaxInt(maxHeight, height)
			maxWidth += width
			columns = append(columns, column)
		}
	}
	s := schema{
		columns: columns,
		width:   maxWidth,
		height:  maxHeight,
	}
	return &s, nil
}

func writeHeaders(f *excelize.File, schema *schema, sheet string, startRow int, startColumn int) error {
	colNum := startColumn
	for _, column := range schema.columns {
		if err := column.printHeaders(f, sheet, startRow, colNum, startRow+schema.height-1); err != nil {
			return err
		}
		width, _ := column.getDimensions()
		colNum += width
	}
	bottomBorder, err := f.NewStyle(`{"border":[{"type":"bottom","style":1}], "font": {"bold": true}}`)
	if err != nil {
		return err
	}
	if err := f.SetCellStyle(sheet, cell(startRow, startColumn),
		cell(startRow+schema.height-1, startColumn+schema.width-1), bottomBorder); err != nil {
		return err
	}
	return nil
}

func writeData(f *excelize.File, columns []*excelColumn, sheet string, startRow int, startColumn int, object interface{}) (int, error) {
	centeredCell, err := f.NewStyle(`{"alignment":{"horizontal":"center"}}`)
	if err != nil {
		return 0, err
	}
	val := reflect.ValueOf(object)
	written := 0
	for _, col := range columns {
		field := val.Field(col.FieldIndex)
		if field.Kind() == reflect.Ptr {
			if !field.Elem().IsValid() {
				width, _ := col.getDimensions()
				written += width
				continue
			}
			field = field.Elem()
		}
		if field.IsZero() && col.NonZero {
			width, _ := col.getDimensions()
			written += width
			continue
		}

		switch field.Kind() {
		case reflect.Int, reflect.Int64, reflect.Int32, reflect.Int16, reflect.Int8:
			if err := f.SetCellInt(sheet, cell(startRow, startColumn+written), int(field.Int())); err != nil {
				return 0, err
			}
			if col.Centered {
				err := f.SetCellStyle(sheet, cell(startRow, startColumn+written), cell(startRow, startColumn+written), centeredCell)
				if err != nil {
					return 0, err
				}
			}
			written++
		case reflect.Uint, reflect.Uint64, reflect.Uint32, reflect.Uint16, reflect.Uint8:
			if err := f.SetCellInt(sheet, cell(startRow, startColumn+written), int(field.Uint())); err != nil {
				return 0, err
			}
			if col.Centered {
				err := f.SetCellStyle(sheet, cell(startRow, startColumn+written), cell(startRow, startColumn+written), centeredCell)
				if err != nil {
					return 0, err
				}
			}
			written++
		case reflect.Float32, reflect.Float64:
			if err := f.SetCellFloat(sheet, cell(startRow, startColumn+written), field.Float(), 2, 64); err != nil {
				return 0, err
			}
			if col.Centered {
				err := f.SetCellStyle(sheet, cell(startRow, startColumn+written), cell(startRow, startColumn+written), centeredCell)
				if err != nil {
					return 0, err
				}
			}
			written++
		case reflect.Bool:
			if err := f.SetCellBool(sheet, cell(startRow, startColumn+written), field.Bool()); err != nil {
				return 0, err
			}
			if col.Centered {
				err := f.SetCellStyle(sheet, cell(startRow, startColumn+written), cell(startRow, startColumn+written), centeredCell)
				if err != nil {
					return 0, err
				}
			}
			written++
		case reflect.Struct:
			nestedWritten, err := writeData(f, col.SubColumns, sheet, startRow, startColumn+written, field.Interface())
			if err != nil {
				return 0, err
			}
			written += nestedWritten
		default:
			if err := f.SetCellStr(sheet, cell(startRow, startColumn+written), fmt.Sprint(field.Interface())); err != nil {
				return 0, err
			}
			if col.HyperlinkFieldIndex != -1 {
				link := val.Field(col.HyperlinkFieldIndex).String()
				if err := f.SetCellHyperLink(sheet, cell(startRow, startColumn+written), link, "External"); err != nil {
					return 0, err
				}
				underlined, err := f.NewStyle(`{"font":{"color":"#1265BE","underline":"single"}}`)
				if err != nil {
					return 0, err
				}
				err = f.SetCellStyle(sheet, cell(startRow, startColumn+written), cell(startRow, startColumn+written), underlined)
				if err != nil {
					return 0, err
				}
			}
			if col.Centered {
				err := f.SetCellStyle(sheet, cell(startRow, startColumn+written), cell(startRow, startColumn+written), centeredCell)
				if err != nil {
					return 0, err
				}
			}
			written++
		}
	}
	return written, nil
}

func col(column int) string {
	cn, _ := excelize.CoordinatesToCellName(column, 1)
	return cn[0 : len(cn)-1]
}

func cell(row int, column int) string {
	name, _ := excelize.CoordinatesToCellName(column, row)
	return name
}

func WriteTable(f *excelize.File, sheet string, startRow int, startColumn int, objects []interface{}) error {
	if len(objects) == 0 {
		return errors.New("no data")
	}
	schema, err := buildSchema(reflect.TypeOf(objects[0]))
	if err != nil {
		return err
	}
	err = writeHeaders(f, schema, sheet, startRow, startColumn)
	if err != nil {
		return err
	}
	for i := 0; i < len(objects); i++ {
		_, err = writeData(f, schema.columns, sheet, startRow+schema.height+i, startColumn, objects[i])
		if err != nil {
			return err
		}
	}
	return nil
}

func (p *ReportPage) Render(f *excelize.File) error {
	f.NewSheet(p.SheetName)
	titleStyle, err := f.NewStyle(`{"font":{"size":30}}`)
	if err != nil {
		return err
	}
	descrStyle, err := f.NewStyle(`{"font":{"size":15}}`)
	if err != nil {
		return err
	}
	err = f.SetCellStr(p.SheetName, cell(1, 1), p.Title)
	if err != nil {
		return err
	}
	err = f.SetCellStyle(p.SheetName, cell(1, 1), cell(1, 1), titleStyle)
	if err != nil {
		return err
	}
	err = f.SetCellStr(p.SheetName, cell(3, 1), p.Description)
	if err != nil {
		return err
	}
	err = f.SetCellStyle(p.SheetName, cell(3, 1), cell(3, 1), descrStyle)
	if err != nil {
		return err
	}
	err = WriteTable(f, p.SheetName, 5, 1, p.Items)
	if err != nil {
		return err
	}
	return nil
}
