package utils

import (
	"bytes"
	"encoding/json"
	"fmt"
	"github.com/valyala/fastjson"
	"io/ioutil"
	"net/http"
	"net/url"
)

type UnexpectedHTTPStatusError struct {
	StatusCode   int
	ResponseBody *http.Response
}

func (e UnexpectedHTTPStatusError) Error() string {
	return fmt.Sprintf("unexpected HTTP status %d, response is %+v", e.StatusCode, e.ResponseBody)
}

func HTTPGet(url string, args url.Values, header http.Header) (*fastjson.Value, error) {
	request, err := http.NewRequest("GET", url, nil)
	if err != nil {
		return nil, fmt.Errorf("error while building http request: %w", err)
	}
	request.URL.RawQuery = args.Encode()
	request.Header = header
	resp, err := http.DefaultClient.Do(request)
	if err != nil {
		return nil, fmt.Errorf("unable to execute http request: %w", err)
	}
	if resp.StatusCode != 200 {
		return nil, UnexpectedHTTPStatusError{
			StatusCode:   resp.StatusCode,
			ResponseBody: resp,
		}
	}
	result, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, fmt.Errorf("unable to read response body: %w", err)
	}
	return fastjson.ParseBytes(result)
}

func HTTPPostForm(url string, values url.Values) (*fastjson.Value, error) {
	resp, err := http.PostForm(url, values)
	if err != nil {
		return nil, fmt.Errorf("error while posting form: %w", err)
	}
	if resp.StatusCode != 200 {
		return nil, UnexpectedHTTPStatusError{
			StatusCode:   resp.StatusCode,
			ResponseBody: resp,
		}
	}
	result, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, fmt.Errorf("unable to read response body: %w", err)
	}
	return fastjson.ParseBytes(result)
}

func HTTPPostJSON(url string, payload interface{}, header http.Header) (*fastjson.Value, error) {
	serialized, err := json.Marshal(payload)
	if err != nil {
		return nil, fmt.Errorf("unable to serialize request: %w", err)
	}
	request, err := http.NewRequest("POST", url, bytes.NewReader(serialized))
	if err != nil {
		return nil, fmt.Errorf("unable to execute http request: %w", err)
	}
	header.Set("Content-Type", "application/json")
	request.Header = header
	resp, err := http.DefaultClient.Do(request)
	if err != nil {
		return nil, fmt.Errorf("unable to execute http request: %w", err)
	}
	if resp.StatusCode != 200 {
		return nil, UnexpectedHTTPStatusError{
			StatusCode:   resp.StatusCode,
			ResponseBody: resp,
		}
	}
	result, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, fmt.Errorf("unable to read response body: %w", err)
	}
	return fastjson.ParseBytes(result)
}

func MinNotZero(a int, b int) int {
	switch {
	case a == 0:
		return b
	case b == 0:
		return a
	case a > b && b != 0:
		return b
	case b > a && a != 0:
		return a
	default:
		return 0
	}
}
