package yql

import (
	"a.yandex-team.ru/travel/hotels/tools/boy_hotels_checker/pkg/utils"
	"errors"
	"fmt"
	"log"
	"net/http"
	"net/url"
	"strconv"
	"time"
)

// Тут можно было бы запилить полноценный клиент c DTO, методами, статусами итп,
// но, кажется, эту работу сделают профессионалы и без нас
// А мы пока по бедности сделаем только то, что нужно для запуска простейших запросов

type YQLStatus string

const yqlRoot = "https://yql.yandex.net/api/v2/operations"

const (
	YQLStatusIdle      YQLStatus = "IDLE"
	YQLStatusPending   YQLStatus = "PENDING"
	YQLStatusRunning   YQLStatus = "RUNNING"
	YQLStatusCompleted YQLStatus = "COMPLETED"
	YQLStatusAborting  YQLStatus = "ABORTING"
	YQLStatusAborted   YQLStatus = "ABORTED"
	YQLStatusError     YQLStatus = "ERROR"
	YQLStatusUnknown   YQLStatus = "UNKNOWN"
)

type YqlOperation struct {
	query   string
	id      string
	token   string
	Status  YQLStatus
	Results []map[string]interface{}
}

type yqlRequest struct {
	Content    string `json:"content"`
	Action     string `json:"action"`
	ActionType string `json:"type"`
}

func NewOperation(query string, token string) *YqlOperation {
	return &YqlOperation{
		query:  query,
		token:  token,
		Status: YQLStatusUnknown,
	}
}

func ExistingOperation(id string, token string) *YqlOperation {
	return &YqlOperation{id: id, token: token}
}

func (o *YqlOperation) Wait() error {
	for o.Status != YQLStatusCompleted && o.Status != YQLStatusAborted && o.Status != YQLStatusError {
		header := http.Header{}
		header.Set("Authorization", fmt.Sprintf("OAuth %s", o.token))
		resp, err := utils.HTTPGet(yqlRoot+"/"+o.id, url.Values{}, header)
		if err != nil {
			return fmt.Errorf("unable to get yql operation: %w", err)
		}
		o.Status = YQLStatus(resp.GetStringBytes("status"))
		if o.Status != YQLStatusCompleted && o.Status != YQLStatusAborted && o.Status != YQLStatusError {
			time.Sleep(10 * time.Second)
		}
	}
	if o.Status == YQLStatusCompleted {
		header := http.Header{}
		header.Set("Authorization", fmt.Sprintf("OAuth %s", o.token))
		args := url.Values{}
		args.Set("filters", "DATA")
		resp, err := utils.HTTPGet(yqlRoot+"/"+o.id+"/results", args, header)
		if err != nil {
			return fmt.Errorf("unable to get yql operation: %w", err)
		}
		results := resp.GetArray("data")
		if len(results) == 0 {
			return nil
		}
		if len(results) > 1 {
			return errors.New("unsupported amount of result sets")
		}
		write := results[0].GetArray("Write")
		if len(write) == 0 {
			return nil
		}
		if len(write) > 1 {
			return errors.New("unsupported amount of result sets in write block")
		}
		typeDef := write[0].Get("Type")
		data := write[0].GetArray("Data")
		if !typeDef.Exists("1", "1") {
			return errors.New("unable to load schema")
		}
		schema := typeDef.GetArray("1", "1")
		fields := make([]string, len(schema))
		for i, o := range schema {
			fields[i] = string(o.GetStringBytes("0"))
		}
		res := make([]map[string]interface{}, len(data))
		for i, datum := range data {
			obj := make(map[string]interface{})
			for j, f := range fields {
				var value interface{}

				typeT := string(schema[j].GetStringBytes("1", "1"))
				if typeT == "" {
					typeT = string(schema[j].GetStringBytes("1", "1", "1"))
				}

				strValue := string(datum.GetStringBytes(strconv.Itoa(j)))
				if strValue == "" {
					strValue = string(datum.GetStringBytes(strconv.Itoa(j), "0"))
				}

				switch typeT {
				case "String", "Utf8", "Json", "Yson", "Uuid":
					value = strValue
				case "Double", "Float", "Decimal":
					value, err = strconv.ParseFloat(strValue, 64)
					if err != nil {
						return fmt.Errorf("unable to parse response: %w", err)
					}
				case "Int8", "Int16", "Int32", "Int64":
					value, err = strconv.ParseInt(strValue, 10, 64)
					if err != nil {
						return fmt.Errorf("unable to parse response: %w", err)
					}
				case "Uint8", "Uint16", "Uint32", "Uint64":
					value, err = strconv.ParseUint(strValue, 10, 64)
					if err != nil {
						return fmt.Errorf("unable to parse response: %w", err)
					}
				default:
					return errors.New("unsupported value type " + typeT)
				}
				obj[f] = value
			}
			res[i] = obj
		}
		o.Results = res
		return nil
	} else {
		return errors.New("unexpected yql operation status " + string(o.Status))
	}
}

func (o *YqlOperation) Start() error {
	if o.query == "" {
		log.Fatal("No query specified")
	}
	if o.id != "" {
		log.Fatal("Query already started")
	}
	header := http.Header{}
	header.Set("Authorization", fmt.Sprintf("OAuth %s", o.token))
	resp, err := utils.HTTPPostJSON(yqlRoot, yqlRequest{
		Content:    o.query,
		Action:     "RUN",
		ActionType: "SQLv1",
	}, header)
	if err != nil {
		return fmt.Errorf("unable to start yql query: %w", err)
	}
	o.id = string(resp.GetStringBytes("id"))
	o.Status = YQLStatus(resp.GetStringBytes("status"))
	return nil
}
