# coding=utf-8

from argparse import ArgumentParser

import datetime
import itertools
import logging
import sys

from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient

from travel.library.python.tools import replace_args_from_env
from travel.hotels.lib.python3.yql import yqllib
from travel.hotels.lib.python3.yt.versioned_path import VersionedPath
from travel.hotels.lib.python3.yt import ytlib


LOG = logging.getLogger(__name__)


class Runner:
    name = 'altay_mappings'
    DATE_FORMAT = '%Y-%m-%d'
    HOTEL_RUBRICS = [
        # Rubric-Id   Rubric Permalink    По-русски   Есть на портале
        30785,       # 184106414          Гостиница        +
        30655,       # 184106316          Общежитие        -
        31632,       # 20699506347        Хостел           +
        30788,       # 184106420          Кемпинг          +
        3501492236,  # 150049871970       Апартаменты      -
        31309,       # 255921949          Отдых на ферме   +
        30781,       # 184106404          Санаторий        +
        30779,       # 184106400          Дом Отдыха       +
        30791,       # 184106426          Турбаза          +
        3501708107,  # 197061821387       Жилье посуточно  -
    ]

    def __init__(self, args):
        self.args = args
        self.yql_client = YqlClient(token=args.yql_token, db=args.yt_proxy)
        self.yt_client = YtClient(proxy=args.yt_proxy, config={'token': args.yt_token})
        self.limit = args.queries_limit
        self.work_path = None

    def run_yql(self, resource_name, parameters=None, format_args=None):
        return yqllib.run_yql_file(self.yql_client, resource_name, 'AltayMappings', parameters=parameters, format_args=format_args)

    def run(self):
        versioned_path = VersionedPath(self.args.yt_path, yt_client=self.yt_client)
        with versioned_path as work_path:
            self.run_at_path(work_path)
            if self.args.transfer_to_cluster:
                # copy results to another cluster
                versioned_path.transfer_results(self.args.transfer_to_cluster, self.args.yt_token, self.args.yt_proxy)

    def get_table_path(self, *parts):
        return ytlib.join(self.work_path, *parts)

    def run_at_path(self, path):
        self.work_path = str(path)
        request = self.run_yql('query_providers.yql')
        providers = []
        for table in request.get_results():
            providers += itertools.chain(*table.rows)

        if self.limit > 0:
            self.run_yql(
                '0_altay_hotel_mappings.yql',
                parameters={
                    '$altay_hotel_mappings_path': self.get_table_path('altay_hotel_mappings'),
                    '$providers_list': list(providers),
                },
                format_args={
                    'temporary_permanence': '    ,\n'.join(
                        ["DictLookup(d, \"{provider}\") AS `{provider}`".format(provider=provider) for provider
                         in providers]
                    )
                },
            )

        if self.limit > 1:
            self.run_yql('1_hotels_permalinks.yql', {
                '$hotels_permalinks_all_path': self.get_table_path('extras', 'hotels_permalinks_all'),
                '$hotels_permalinks_published_path': self.get_table_path('hotels_permalinks_published'),
                '$providers_list': list(providers),
                '$hotel_rubrics': self.HOTEL_RUBRICS,
            })

        if self.limit > 2:
            self.run_yql('2_permalink_to_cluster_permalink.yql', {
                '$hotels_permalinks_published_path': self.get_table_path('hotels_permalinks_published'),
                '$hotels_permalinks_all_path': self.get_table_path('extras', 'hotels_permalinks_all'),
                '$permalink_to_cluster_permalink_path': self.get_table_path('permalink_to_cluster_permalink'),
                '$permalink_to_cluster_permalink_filtered_path': self.get_table_path('extras', 'permalink_to_cluster_permalink_filtered'),
            })

        if self.limit > 3:
            self.run_yql('3_permalink_to_partnerid_originalid.yql', {
                '$permalink_to_cluster_permalink_path': self.get_table_path('permalink_to_cluster_permalink'),
                '$permalink_to_partnerid_originalid_path': self.get_table_path('permalink_to_partnerid_originalid'),
                '$partnerid_originalid_to_cluster_permalink_path': self.get_table_path('partnerid_originalid_to_cluster_permalink'),
            })

        if self.limit > 4:
            self.run_yql('4_permalink_to_hotel_info.yql', {
                '$hotels_permalinks_all_path': self.get_table_path('extras', 'hotels_permalinks_all'),
                '$hotel_rubrics': self.HOTEL_RUBRICS,
                '$permalink_to_cluster_permalink_path': self.get_table_path('permalink_to_cluster_permalink'),
                '$permalink_to_hotel_info_path': self.get_table_path('permalink_to_hotel_info'),
                '$permalink_to_partnerid_originalid_path': self.get_table_path('permalink_to_partnerid_originalid'),
            })

        if self.limit > 5:
            searcher_logs_startdate = (datetime.datetime.now() - datetime.timedelta(days=30)).strftime(self.DATE_FORMAT)
            self.run_yql('5_data_issues.yql', {
                '$hotels_permalinks_all': self.get_table_path('extras', 'hotels_permalinks_all'),
                '$searcher_logs': "//home/logfeller/logs/travel-hotels-search-result-log/1d",
                '$partnerid_originalid_to_cluster_permalink': self.get_table_path("partnerid_originalid_to_cluster_permalink"),
                '$permalink_to_cluster_permalink': self.get_table_path('permalink_to_cluster_permalink'),
                '$permalink_to_hotel_info': self.get_table_path('permalink_to_hotel_info'),
                '$searcher_logs_startdate': searcher_logs_startdate,
                '$hotel_rubrics': self.HOTEL_RUBRICS,
                '$output_dir': self.get_table_path("data_issues/"),
            })


def main():
    logging.basicConfig(level=logging.INFO, format="%(asctime)-15s | %(module)s | %(levelname)s | %(message)s", stream=sys.stdout)
    logging.getLogger('yt.packages.urllib3.connectionpool').setLevel(logging.WARNING)

    parser = ArgumentParser()
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--yt-path', required=True)
    parser.add_argument('--yql-token', required=True)
    parser.add_argument("--transfer-to-cluster", default=None)
    parser.add_argument("--debug", action='store_true')
    parser.add_argument("--queries-limit", default=100, type=int, help='Allows to run only {limit} first steps for debugging purposes')
    args = parser.parse_args(args=replace_args_from_env())
    Runner(args).run()


if __name__ == '__main__':
    main()
