# coding=utf-8
from __future__ import print_function

import csv
import io
import logging

from library.python import resource
from travel.hotels.lib.python import yqllib
from yql.api.v1.client import YqlClient
from yt.transfer_manager.client import TransferManager
import yt.wrapper as yt

from travel.hotels.lib.python.versioned_process import VersionedProcess

LOG = logging.getLogger(__name__)


class GeocounterDataBuilder(VersionedProcess):
    name = 'geocounter_data'
    DATE_FORMAT = '%Y-%m-%d'

    def __init__(self, session, args):
        super(GeocounterDataBuilder, self).__init__(session, args)
        self.yql_client = YqlClient(token=args.yql_token, db=args.yt_proxy)
        self.yt_proxy = args.yt_proxy
        self.yt_client = yt.client.YtClient(proxy=args.yt_proxy, token=args.yt_token)

        self.export_results = args.export_results
        if self.export_results:
            self.transfer_manager = TransferManager(token=args.yt_token)
            self.export_destination = args.export_destination
            self.export_client = yt.client.YtClient(proxy=self.export_destination, token=args.yt_token)

    def run_query(self, query, parameters=None, debug=None):
        if debug is None:
            debug = self.debug
        return yqllib.run_query(query=query, parameters=parameters, client=self.yql_client, debug=debug,
                                syntax_version=1)

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        common_group = parser.add_argument_group(GeocounterDataBuilder.name + " - common")
        common_group.add_argument("--yql-token", default=None)
        common_group.add_argument("--export-results", action='store_true')
        common_group.add_argument("--export-destination", default='arnold')

    def run(self):
        custom_bbox_table_path = self.get_table_path('custom_bbox')
        self._upload_custom_bbox(custom_bbox_table_path)
        geocounter_table_query = resource.find('geocounter_table.yql').decode('utf-8')
        geocounter_table_request = self.run_query(geocounter_table_query, {
            '$geocounter_table_path': self.get_table_path('geocounter_table'),
        })

        regions_table_query = resource.find('regions_table.yql').decode('utf-8')
        regions_table_request = self.run_query(regions_table_query, {
            '$output_path': self.get_table_path('regions_table'),
            '$logs_days': 28,
            '$custom_bbox_path': custom_bbox_table_path
        })

        yqllib.wait_results(geocounter_table_request)
        yqllib.wait_results(regions_table_request)

        # copy results to another cluster
        if self.export_results and not self.debug:
            self.export_client.create('map_node', self.get_run_dir(), recursive=True)
            LOG.info("Exporting to {}".format(self.export_destination))
            export_tasks = self.transfer_manager.add_tasks(
                source_cluster=self.yt_proxy,
                source_pattern=self.get_run_dir(),
                destination_cluster=self.export_destination,
                destination_pattern=self.get_run_dir(),
                enable_failed_tasks_restarting=True,
                include_files=True,
                sync=True
            )
            LOG.info("Exporting results to {}, created tasks: {}".format(self.export_destination, export_tasks))
            self.export_client.link(self.get_run_dir(), self.get_latest_path(), force=True)

    def _upload_custom_bbox(self, synonyms_table):
        self._upload_csv(synonyms_table, 'custom_bbox.csv', {
            'geoId': 'int32',
            'lowerLeftLat': 'double',
            'lowerLeftLon': 'double',
            'upperRightLat': 'double',
            'upperRightLon': 'double',
        })

    def schema_from_dict(self, schema):
        return [{'name': k, 'type': v} for k, v in schema.items()]

    def _upload_csv(self, table_path, resource_name, schema):
        self.yt_client.create("table", table_path, attributes={"schema": self.schema_from_dict(schema)})
        converters = {
            'int32': int,
            'double': float,
        }
        csv_data = resource.find(resource_name).decode('utf-8')
        rows = []
        for line in csv.DictReader(io.StringIO(csv_data), delimiter=',', quotechar='"'):
            row = dict()
            for field, yt_type in schema.items():
                converter = converters.get(yt_type)
                if converter is None:
                    raise Exception("Cannot find convertor for field {}, type is {}".format(field, yt_type))
                row[field] = converter(line[field])
            rows.append(row)
        self.yt_client.write_table(table_path, rows)


if __name__ == '__main__':
    GeocounterDataBuilder.main()
