from __future__ import print_function

import argparse
import datetime
import logging
import pprint
import socket
import sys
import uuid
import time
import requests

import grpc
from google.protobuf import json_format, text_format, timestamp_pb2

import travel.proto.commons_pb2 as commons_pb2
import travel.hotels.proto2.hotels_pb2 as hotels_pb2
import travel.hotels.proto.offer_invalidation.event_pb2 as invalidation_event_pb2
import travel.hotels.proto.offer_invalidation.offer_invalidation_service_pb2 as offer_invalidation_service_pb2
import travel.hotels.proto.offer_invalidation.offer_invalidation_service_pb2_grpc as offer_invalidation_service_pb2_grpc
import travel.hotels.proto.offercache_api.aux_pb2 as oc_api_aux_pb2

OFFERCACHE_TESTING_URL = 'http://travel-hotels-offercache-test.yandex.net:11001'
OFFERCACHE_PROD_URL = 'http://travel-hotels-offercache.yandex.net:11001'


class OfferCacheClient:
    def __init__(self, base_url: str):
        self.base_url = base_url

    def get_partner_hotel_ids(self, permalink: int):
        resp = requests.get(f'{self.base_url}/permalink', params={'Permalink': permalink})
        message = json_format.Parse(resp.text, oc_api_aux_pb2.TPermalinkResp())
        return [(x.PartnerId, x.OriginalId) for x in message.PartnerHotels]


class OfferInvalidationServiceClient:
    def __init__(self, host: str, port: int, dry_run: bool):
        self.host = host
        self.port = port
        self.dry_run = dry_run
        self.stub = offer_invalidation_service_pb2_grpc.OfferInvalidationServiceV1Stub(grpc.insecure_channel(f'{host}:{port}'))

    def invalidate_by_checkin_checkout(self, partner_id: hotels_pb2.EPartnerId, original_id: str, checkin: str, checkout: str, timestamp: int = None):
        logging.info(f'Invalidating by checkin/out, partner_id={partner_id}, original_id={original_id}, checkin={checkin}, checkout={checkout}.')
        invalidation_filter = invalidation_event_pb2.TFilter(
            CheckInCheckOutFilter=invalidation_event_pb2.TCheckInCheckOutFilter(
                CheckInDate=checkin,
                CheckOutDate=checkout
            )
        )
        self.send_invalidation_request(partner_id, original_id, invalidation_filter, timestamp)

    def invalidate_by_target_date(self, partner_id: hotels_pb2.EPartnerId, original_id: str, from_date_inclusive: str, to_date_inclusive: str, timestamp: int = None):
        logging.info(f'Invalidating by target date, partner_id={partner_id}, original_id={original_id}, from_date_inclusive={from_date_inclusive}, to_date_inclusive={to_date_inclusive}.')
        invalidation_filter = invalidation_event_pb2.TFilter(
            TargetIntervalFilter=invalidation_event_pb2.TTargetIntervalFilter(
                DateFromInclusive=from_date_inclusive,
                DateToInclusive=to_date_inclusive
            )
        )
        self.send_invalidation_request(partner_id, original_id, invalidation_filter, timestamp)

    def send_invalidation_request(self, partner_id: hotels_pb2.EPartnerId, original_id: str, invalidation_filter: invalidation_event_pb2.TFilter, timestamp: int = None):
        if timestamp is None:
            timestamp = int(time.time())

        pb_timestamp = timestamp_pb2.Timestamp()
        pb_timestamp.FromSeconds(timestamp)
        req = offer_invalidation_service_pb2.TOfferInvalidationReq(
            HotelId=hotels_pb2.THotelId(
                PartnerId=partner_id,
                OriginalId=original_id
            ),
            Currency=commons_pb2.C_RUB,
            Timestamp=pb_timestamp,
            Filters=[invalidation_filter],
            OfferInvalidationSource=invalidation_event_pb2.OIS_MANUAL,
        )
        metadata = self._create_metadata()
        logging.info(f'Sending invalidation event to {self.host}:{self.port}...')
        logging.debug(f'-> REQ {req.__class__}\n{self._protobuf_to_string(req)}\n{pprint.pformat(metadata)}')

        if not self.dry_run:
            rsp = self.stub.InvalidateOffers(req, metadata=metadata.items())
            logging.debug(f'<- RSP {rsp.__class__}\n{self._protobuf_to_string(rsp)}')

        logging.info('Done')

    def _protobuf_to_string(self, message):
        return text_format.MessageToString(message, as_utf8=True)

    def _create_metadata(self):
        return {
            "ya-grpc-call-id": str(uuid.uuid1()),
            "ya-grpc-started-at": datetime.datetime.utcnow().isoformat() + "Z",
            "ya-grpc-timeout-msec": str(60000),
            "ya-grpc-fqdn": socket.getfqdn(),
        }


def invalidate(args, do_invalidate):
    logging.info(f'Dry run is {args.dry_run}')

    if args.permalink is None and args.hotel_id is None:
        raise Exception('Either permalink or hotel_id is required')

    if args.permalink is not None and args.hotel_id is not None:
        raise Exception('Only one of permalink and hotel_id is required')

    client = OfferInvalidationServiceClient(args.host, args.port, args.dry_run)

    if args.hotel_id is None:
        if args.offercache_env == 'testing':
            oc_url = OFFERCACHE_TESTING_URL
        elif args.offercache_env == 'prod':
            oc_url = OFFERCACHE_PROD_URL
        else:
            raise ValueError(f'Unknown env: {args.offercache_env}')
        oc_client = OfferCacheClient(oc_url)
        for partner_id, original_id in oc_client.get_partner_hotel_ids(args.permalink):
            do_invalidate(client, partner_id, original_id)
    else:
        partner_id_raw, original_id = args.hotel_id.split('.')
        partner_id = hotels_pb2.EPartnerId.Value(partner_id_raw)
        do_invalidate(client, partner_id, original_id)


def is_valid_date(date):
    if date is None:
        return True
    try:
        datetime.datetime.strptime(date, '%Y-%m-%d')
        return True
    except ValueError:
        return False


def invalidate_by_checkin_checkout(args):
    if not is_valid_date(args.checkin) or not is_valid_date(args.checkout):
        raise Exception('Expected date in format YYYY-MM-DD')

    def do_invalidate(client, partner_id, original_id):
        client.invalidate_by_checkin_checkout(partner_id, original_id, args.checkin, args.checkout)

    invalidate(args, do_invalidate)


def invalidate_by_target_date(args):
    if not is_valid_date(args.from_date_inclusive) or not is_valid_date(args.to_date_inclusive):
        raise Exception('Expected date in format YYYY-MM-DD')

    def do_invalidate(client, partner_id, original_id):
        client.invalidate_by_target_date(partner_id, original_id, args.from_date_inclusive, args.to_date_inclusive)

    invalidate(args, do_invalidate)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--host", default="travel-hotels-busbroker.yandex.net")
    parser.add_argument("--port", default=15003)
    parser.add_argument("--dry-run", type=bool, default=False)
    parser.add_argument("--verbose", action='store_true', default=False)
    parser.add_argument("--offercache-env", help="testing|prod", default="prod")
    parser.add_argument('--permalink', type=int)
    parser.add_argument('--hotel-id', help='In format partnerid.originalid (PI_BOOKING.123)')
    subparsers = parser.add_subparsers(dest='cmd')
    subparsers.required = True

    by_inout = subparsers.add_parser("by-inout")
    by_inout.set_defaults(func=invalidate_by_checkin_checkout)
    by_inout.add_argument("--checkin")
    by_inout.add_argument("--checkout")

    by_target = subparsers.add_parser("by-target")
    by_target.set_defaults(func=invalidate_by_target_date)
    by_target.add_argument("--from-date-inclusive")
    by_target.add_argument("--to-date-inclusive")

    args = parser.parse_args()
    logging.basicConfig(level=(logging.DEBUG if args.verbose else logging.INFO),
                        format="%(asctime)-15s | %(levelname)s | %(message)s",
                        stream=sys.stdout)

    args.func(args)


if __name__ == "__main__":
    main()
