# -*- coding: utf-8 -*-

from argparse import ArgumentParser
import logging
import sys
import datetime
import calendar

import yaml
from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient
import yt.wrapper as yt
from library.python import resource
import statface_client

from travel.library.python.tools import replace_args_from_env
from startrek_client import Startrek
from jinja2 import Template

from travel.hotels.lib.python import yqllib


class Runner(object):
    SCALES = ['daily', 'weekly', 'monthly']

    SCALE_START_DATE = {
        'daily': lambda d: d,
        'weekly': lambda d: d - datetime.timedelta(days=d.weekday()),
        'monthly': lambda d: datetime.date(d.year, d.month, 1),
    }

    SCALE_END_DATE = {
        'daily': lambda d: d,
        'weekly': lambda d: d + datetime.timedelta(days=6 - d.weekday()),
        'monthly': lambda d: datetime.date(d.year, d.month, calendar.monthrange(d.year, d.month)[1]),
    }

    def __init__(self, args, date):
        yql_args = {
            'db': args.yt_proxy,
            'token': args.yql_token
        }
        self.yql_client = YqlClient(**yql_args)

        def module_filter(mod):
            if 'hashlib' in getattr(mod, '__name__', ''):
                return False
            if 'weakref' in getattr(mod, '__name__', ''):
                return False
            return True
        yt_config = {
            'pickling': {"module_filter": module_filter},
            'token': args.yt_token
        }
        self.yt_client = YtClient(proxy=args.yt_proxy, config=yt_config)
        self.date = date
        self.date_str = date.isoformat()
        self.args = args
        self.yt_temp_dir = yt.ypath_join(args.yt_path, 'temp', self.date_str)
        # TODO use <date> instead of latest
        self.table_altay_hotel_info = '//home/travel/prod/general/altay_mappings/latest/permalink_to_hotel_info'
        self.table_altay_orig_ids = '//home/travel/prod/general/altay_mappings/latest/permalink_to_partnerid_originalid'
        self.table_altay_clusters = '//home/travel/prod/general/altay_mappings/latest/permalink_to_cluster_permalink'
        self.table_catroom_permalinks = yt.ypath_join(self.yt_temp_dir, 'catroom_permalinks')
        self.table_catroom_permalinks_enriched = yt.ypath_join(self.yt_temp_dir, 'catroom_permalinks_enriched')
        self.table_permalink_stats_offercache = yt.ypath_join(self.yt_temp_dir, 'permalink_stats_offercache')
        self.table_permalink_stats_catroom_by_op = yt.ypath_join(self.yt_temp_dir, 'permalink_stats_catroom_by_op')
        self.table_permalink_stats_catroom_by_partner = yt.ypath_join(self.yt_temp_dir, 'permalink_stats_catroom_by_partner')
        self.table_join_results_intermediate = yt.ypath_join(self.yt_temp_dir, 'join_results_intermediate')
        self.table_regions = yt.ypath_join(self.yt_temp_dir, 'regions')
        self.yt_scale_dirs = {scale: yt.ypath_join(args.yt_path, scale) for scale in self.SCALES}
        self.start_date_strs = {scale: self.SCALE_START_DATE[scale](self.date).isoformat() for scale in self.SCALES}
        self.end_date_strs = {scale: self.SCALE_END_DATE[scale](self.date).isoformat() for scale in self.SCALES}
        self.table_results = {scale: yt.ypath_join(self.yt_scale_dirs[scale], self.start_date_strs[scale]) for scale in self.SCALES}

    def run(self):
        self._ensure_yt_dir(self.yt_temp_dir)
        for scale in self.SCALES:
            self._ensure_yt_dir(self.yt_scale_dirs[scale])
            if scale == 'daily':
                self.step_1_get_catroomed_permalinks()
                self.step_2_enrich_catroomed_permalinks()
                self.step_3_parse_offercache_logs()
                self.step_4_parse_altay_mappings()
                self.step_5_parse_catroom_logs()
                self.step_6_join_stats()
                self.upload_to_startrek()
            else:
                if not self.args.no_upscale:
                    self.step_7_upscale(scale)
            self.upload_to_stat(scale)
        logging.info("Deleting temporary directory %s" % self.yt_temp_dir)
        self.yt_client.remove(self.yt_temp_dir, recursive=True)

    def _ensure_yt_dir(self, yt_path):
        self.yt_client.create('map_node', yt_path, recursive=True, ignore_existing=True)

    def _exec_yql(self, query_name, params=None):
        query = resource.find(query_name)
        if query is None:
            raise Exception("Resource not found by name '%s'" % query_name)
        yqllib.wait_results(yqllib.run_query(query.decode('utf-8'), parameters=params, syntax_version=1,
                                             title='YQL:CatRoomStat: %s' % query_name))

    def step_1_get_catroomed_permalinks(self):
        logging.info("Getting CatRoom permalinks")
        self._exec_yql('1_get_catroomed_permalinks.yql', {
            '$input_path_catroom_mappings': self.args.catroom_export_mappings_table,
            '$output_path': self.table_catroom_permalinks,
        })


    def step_2_enrich_catroomed_permalinks(self):
        logging.info("Enriching catroomed permalinks with clusters")
        self._exec_yql('2_enrich_catroom_data.yql', {
            '$input_path_catroom_permalinks': self.table_catroom_permalinks,
            '$input_path_altay_clusters': self.table_altay_clusters,
            '$output_path_catroom_permalinks_enriched': self.table_catroom_permalinks_enriched,
        })

    def step_3_parse_offercache_logs(self):
        logging.info("Processing OfferCache logs for date %s" % self.date)
        self._exec_yql('3_parse_offercache_logs.yql', {
            '$input_path_offercache_log': "//home/logfeller/logs/travel-hotels-offercache-log/1d/%s" % self.date_str,
            '$input_path_altay_clusters': self.table_altay_clusters,
            '$input_path_catroom_permalinks': self.table_catroom_permalinks_enriched,
            '$output_path_permalink_stats_offercache': self.table_permalink_stats_offercache
        })

    def step_4_parse_altay_mappings(self):
        logging.info("Parse altay mappings")
        self._exec_yql('4_parse_altay_mappings.yql', {
            '$input_path_altay_mappings': self.table_altay_hotel_info,
            '$output_path_regions': self.table_regions
        })

    def step_5_parse_catroom_logs(self):
        logging.info("Parse CatRoom logs")
        self._exec_yql('5_parse_catroom_logs.yql', {
            '$input_path_catroom_log': "//home/logfeller/logs/travel-prod-hotels-catroom-eventlog/1d/%s" % self.date_str,
            '$input_path_altay_clusters': self.table_altay_clusters,
            '$input_path_catroom_permalinks': self.table_catroom_permalinks_enriched,
            '$output_path_permalink_stats_catroom_by_partner': self.table_permalink_stats_catroom_by_partner,
            '$output_path_permalink_stats_catroom_by_op': self.table_permalink_stats_catroom_by_op,
        })

    def step_6_join_stats(self):
        logging.info("Joining stat to single table")
        self._exec_yql('6_join_stats.yql', {
            '$input_path_permalink_stats_offercache': self.table_permalink_stats_offercache,
            '$input_path_permalink_stats_catroom_by_op': self.table_permalink_stats_catroom_by_op,
            '$input_path_altay_mappings': self.table_altay_hotel_info,
            '$input_path_altay_permalink_to_original_id': self.table_altay_orig_ids,
            '$input_path_regions': self.table_regions,
            '$input_date': self.date_str,
            '$output_path_intermediate': self.table_join_results_intermediate,
            '$output_path': self.table_results['daily'],
        })

    def step_7_upscale(self, scale):
        logging.info("Upscale stat to scale %s [%s - %s]" % (scale, self.start_date_strs[scale], self.end_date_strs[scale]))
        self._exec_yql('7_upscale.yql', {
            '$input_path_prefix': self.yt_scale_dirs['daily'],
            '$input_path_min': self.start_date_strs[scale],
            '$input_path_max': self.end_date_strs[scale],
            '$input_date': self.start_date_strs[scale],
            '$output_path': self.table_results[scale]
        })

    def upload_to_stat(self, scale):
        if not self.args.stat_token:
            return
        client = statface_client.StatfaceClient(host='upload.stat.yandex-team.ru', oauth_token=self.args.stat_token)
        report_path = 'Travel/CatRoom/main'
        report = client.get_report(report_path)
        report_config_dict = yaml.load(resource.find('report_config.yaml'))
        logging.info('Uploading report config to scale %s' % scale)
        config_object = statface_client.StatfaceReportConfig()
        config_object.update_from_dict(report_config_dict)
        config_object.title = 'Статистика CatRoom ({})'.format(datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S'))
        report.upload_config(config_object)
        logging.info('Uploading report data to scale %s' % scale)
        report.upload_yt_data(scale=scale, cluster=self.args.yt_proxy, table_path=self.table_results[scale],
                              yt_token=self.args.yt_token)

    def upload_to_startrek(self):
        if not self.args.st_token:
            return
        comment_template = Template(resource.find('st_comment.txt').decode('utf-8'))
        unknown_subhotels = list()
        for row in self.yt_client.read_table(self.table_permalink_stats_catroom_by_partner):
            unknown_subhotels.append({
                'permalink': row['permalink'],
                'partner_id': row['partner_id'],
                'orig_hotel_id': row['orig_hotel_id'],
                'count': row['count']
            })
        unknown_subhotels = sorted(unknown_subhotels, key=lambda item: (item['permalink'], -item['count']))
        txt = comment_template.render(unknown_subhotels=unknown_subhotels)
        client = Startrek(useragent='python', base_url='https://st-api.yandex-team.ru', token=self.args.st_token)
        if self.args.st_issue_hotels:
            issue = client.issues[self.args.st_issue_hotels]
            issue.comments.create(text=txt)
        if self.args.st_issue_acontent and self.date.weekday() == self.args.st_issue_acontent_weekday:
            issue = client.issues.create(
                queue=self.args.st_issue_acontent.split('-')[0],
                summary='Неизвестные партнёрские отели ' + self.date_str,
                type={'name': 'Task'},
                description=txt,
                assignee=self.args.st_issue_acontent_assignee,
                followers=['alexcrush', 'ritapak'],
                components=['travel'],
            )
            issue.links.create(issue=self.args.st_issue_acontent, relationship='is_subtask_for')


def main():
    logging.basicConfig(level=logging.DEBUG, format="%(asctime)-15s | %(module)s | %(levelname)s | %(message)s", stream=sys.stdout)
    logging.getLogger('yt.packages.urllib3.connectionpool').setLevel(logging.WARNING)

    def parse_date(s):
        assert len(s) == 10
        assert s[4] == s[7] == '-'
        return datetime.date(year=int(s[0:4]), month=int(s[5:7]), day=int(s[8:10]))

    parser = ArgumentParser()
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--yql-token', required=True)
    parser.add_argument('--stat-token')
    parser.add_argument('--st-token')
    parser.add_argument('--st-issue-hotels')
    parser.add_argument('--st-issue-acontent')
    parser.add_argument('--st-issue-acontent-weekday', type=int, default=5)
    parser.add_argument('--st-issue-acontent-assignee')
    parser.add_argument('--yt-path', required=True)
    parser.add_argument('--catroom-export-mappings-table', default='//home/travel/prod/content_manager/catroom/export/catroom/mappings')
    parser.add_argument('--date-from', default=(datetime.date.today() + datetime.timedelta(days=-1)).isoformat(), type=parse_date)
    parser.add_argument('--date-to', default=(datetime.date.today() + datetime.timedelta(days=-1)).isoformat(), type=parse_date)
    parser.add_argument('--no-upscale', action='store_true')
    args = parser.parse_args(args=replace_args_from_env())
    date = args.date_from
    while date <= args.date_to:
        logging.info("Processing date %s" % date)
        Runner(args, date).run()
        date += datetime.timedelta(days=1)


if __name__ == '__main__':
    main()
