# -*- coding: utf-8 -*-

from argparse import ArgumentParser
from collections import Counter
from datetime import datetime
from enum import auto, Enum
from typing import Dict, Optional
import logging

# noinspection PyUnresolvedReferences
from parse import parse

from travel.library.python.yandex_vault import resolve_secrets_ns
from travel.hotels.tools.check_nirvana_logs.hitman_client import HitmanClient
from travel.hotels.tools.check_nirvana_logs.nirvana_client import NirvanaClient
from travel.hotels.tools.check_nirvana_logs.oauth_client import OauthClient


class Error(Enum):
    NO_STDERR_LOG = auto()
    NO_STDOUT_LOG = auto()
    EXIT_STATUS_127 = auto()
    SEGFAULT = auto()
    SOLOMON_404 = auto()
    SOLOMON_502 = auto()
    STATFACE_INTERNAL = auto()
    STOCKS_CONNECTION = auto()
    TVM_NTP = auto()
    XML_SYNTAX = auto()
    YAV_CONNECTION = auto()
    YT_EXPIRED_TRANSACTION = auto()
    YT_MEMORY_LIMIT = auto()
    YT_NO_OPERATION = auto()
    YT_PROXY_UNAVAILABLE = auto()
    YT_READ_TIMEOUT = auto()
    UNKNOWN = auto()


ErrorMapping = Dict[Error, str]

STDERR_MAPPING: ErrorMapping = {
    Error.EXIT_STATUS_127: 'returned non-zero exit status 127',
    Error.SEGFAULT: 'Segmentation fault (core dumped)',
    Error.SOLOMON_404: 'HTTPError: 404 Client Error: Not Found for url: https://solomon.yandex.net',
    Error.SOLOMON_502: 'HTTPError: 502 Server Error: Bad Gateway for url: https://solomon.yandex.net',
    Error.STOCKS_CONNECTION: "ConnectionError: HTTPConnectionPool(host='stocks.yandex.net', port=80): Max retries",
    Error.STATFACE_INTERNAL: 'StatfaceHttpResponseRetriableError: Request to statface failed (retriable):',
    Error.TVM_NTP: r'Please, fix your NTP: https:\/\/wiki.yandex-team.ru\/passport\/tvm2\/400',
    Error.XML_SYNTAX: 'XMLSyntaxError',
    Error.YT_EXPIRED_TRANSACTION: 'has expired or was aborted',
    Error.YT_MEMORY_LIMIT: 'Memory limit exceeded',
    Error.YT_NO_OPERATION: 'YtHttpResponseError: No such operation',
    Error.YT_PROXY_UNAVAILABLE: 'Proxy is unavailable',
}


STDOUT_MAPPING: ErrorMapping = {
    Error.YAV_CONNECTION: 'Failed to resolve secrets',
    Error.YT_READ_TIMEOUT: 'yt.packages.requests.exceptions.ReadTimeout',

}


class LogsChecker:

    def __init__(self, hitman_token: str, nirvana_token: str, date_from: datetime) -> None:
        self.hitman_token = hitman_token
        self.nirvana_token = nirvana_token
        self.date_from = date_from

        self.session = OauthClient.get_session()

    @staticmethod
    def get_error_from_mapping(text: str, mapping: Dict[Error, str]) -> Optional[Error]:
        for error, substring in mapping.items():
            if substring in text:
                return error

    def get_log_error(
            self,
            logs: Dict[str, str],
            log_name: str,
            mapping: ErrorMapping,
            no_log_error: Error
    ) -> Optional[Error]:
        if not log_name:
            return
        url = logs[log_name]
        rsp = self.session.get(url)

        if rsp.status_code == 404:
            logging.info(f'{no_log_error} {url}')
            return no_log_error
        rsp.raise_for_status()

        return self.get_error_from_mapping(rsp.text, mapping)

    def get_error_type(self, logs) -> Error:
        stdout_log_name = None
        stderr_log_name = None
        for log_name in logs.keys():
            if 'stdout' in log_name and 'launcher' not in log_name:
                stdout_log_name = log_name
            if 'stderr' in log_name:
                stderr_log_name = log_name

        error = self.get_log_error(logs, stdout_log_name, STDOUT_MAPPING, Error.NO_STDOUT_LOG)
        if error:
            return error

        error = self.get_log_error(logs, stderr_log_name, STDERR_MAPPING, Error.NO_STDERR_LOG)
        if error:
            return error

        if stderr_log_name:
            logging.info(f'stderr {logs[stderr_log_name]}')
        if stdout_log_name:
            logging.info(f'stdout {logs[stdout_log_name]}')
        return Error.UNKNOWN

    def check(self) -> None:
        nc = NirvanaClient(token=self.nirvana_token)
        hc = HitmanClient(token=self.hitman_token)

        failed_total = 0
        process_counter = Counter()
        errors_counter = Counter()

        processes = hc.get_project_processes('travel-cpa-test')
        logging.info(f'Got {len(processes)} processes')
        for index, process in enumerate(processes):
            if process['state'] != 'ENABLED':
                continue

            process_code = process['code']
            logging.info(f'{process_code}: {index} of {len(processes)}')

            for job in hc.get_process_jobs(process_code, 'FAILED', date_from=self.date_from):
                assert process_code == job['processCode']
                for execution in job['executions']:
                    workflow_uid = execution['workflow']

                    logs = nc.get_workflow_logs(workflow_uid)
                    if not logs:
                        continue

                    error_type = self.get_error_type(logs)

                    failed_total += 1
                    process_counter[process_code] += 1
                    errors_counter[error_type] += 1

        logging.info(failed_total)

        logging.info('process_counter')
        keys = sorted(process_counter.keys(), key=lambda x: process_counter[x], reverse=True)
        for key in keys:
            logging.info(f'{key}\t{process_counter[key]}')

        logging.info('errors_counter')
        keys = sorted(errors_counter.keys(), key=lambda x: errors_counter[x], reverse=True)
        for key in keys:
            logging.info(f'{key}\t{errors_counter[key]}')


def parse_datetime(dt_text: str) -> datetime:
    return parse('{:ti}', dt_text)[0]


def main():
    logging.basicConfig(level=logging.INFO)

    parser = ArgumentParser()
    parser.add_argument('--vault-token')

    parser.add_argument('--hitman-token', required=True)
    parser.add_argument('--nirvana-token', required=True)
    parser.add_argument('--date-from', default=None, type=parse_datetime)

    options = parser.parse_args()
    logging.info(f'Working with: {options}')
    resolve_secrets_ns(options)

    checker = LogsChecker(
        hitman_token=options.hitman_token,
        nirvana_token=options.nirvana_token,
        date_from=options.date_from,
    )
    checker.check()


if __name__ == '__main__':
    main()
