import logging
import sys
from argparse import ArgumentParser, Namespace
from inspect import isclass

from sandbox.common import auth
from sandbox.common import rest
from sandbox.projects.Travel import resources
from sandbox.sdk2 import Resource

from travel.hotels.tools.clean_sandbox_resources.resource_info import ResourceInfo
from travel.library.python.tools import replace_args_from_env


class Runner:

    def __init__(self, args: Namespace):
        self.args = args
        self.sb_client = rest.Client(auth=auth.OAuth(args.sb_token))
        self.additional_resources = args.additional_resources or []
        self.untouchable_resources = set(args.untouchable_resources or [])

    def run(self) -> None:
        for key, value in resources.__dict__.items():
            if not isclass(value) or not issubclass(value, Resource):
                continue
            if key in self.untouchable_resources:
                logging.info(f'Resource {key} is untouchable. Skipping it')
                continue
            self._process_resources(key)
        for resource_type in self.additional_resources:
            self._process_resources(resource_type)
        logging.info('All done')

    def _process_resources(self, resource_type: str) -> None:
        logging.info(resource_type)
        resources_grouped = dict()
        resources_all = list()

        resp = self.sb_client.resource.read(
            type=resource_type,
            owner='TRAVEL',
            state='READY',
            limit=self.args.sb_request_limit,
            attrs={"ttl": "inf"},
        )
        for r in resp['items']:
            resource_info = ResourceInfo(
                id=r['id'],
                created_at=r['time']['created'],
                released_to=r['attributes'].get('released'),
            )
            resources_group = resources_grouped.setdefault(resource_info.released_to, list())
            resources_group.append(resource_info)
            resources_all.append(resource_info)

        for released_to, resources_group in resources_grouped.items():
            logging.info(f'Released to {released_to}')
            resources_by_creation = sorted(resources_group, key=lambda x: x.created_at, reverse=True)

            resources_to_keep = resources_by_creation[:self.args.resources_to_keep]
            for r in resources_to_keep:
                logging.info(f'Resource to keep {r}')

            resources_to_kill = resources_by_creation[self.args.resources_to_keep:]
            killed_ids = list()
            for r in resources_to_kill:
                logging.info(f'Resource to be killed {r}')
                if not self.args.dry_run:
                    self._kill_resource(r)
                    killed_ids.append(r.id)

            if killed_ids:
                logging.info('Requesting resource deleting')
                self.sb_client.batch.resources["delete"].update(
                    id=killed_ids,
                    comment='Remove outdated resources',
                )

    def _kill_resource(self, r: ResourceInfo) -> None:
        self.sb_client.resource[r.id].attribute['ttl'] = {'value': self.args.sb_ttl}


def main():
    logging.basicConfig(
        level=logging.INFO,
        format="%(asctime)-15s | %(module)s | %(levelname)s | %(message)s",
        stream=sys.stdout,
    )

    parser = ArgumentParser()
    parser.add_argument('--sb-token', required=True)
    parser.add_argument('--sb-request-limit', type=int, default=1000)
    parser.add_argument('--resources-to-keep', type=int, default=10)
    parser.add_argument('--additional-resources', nargs='*')
    parser.add_argument('--untouchable-resources', nargs='*')
    parser.add_argument('--sb-ttl', default='1')
    parser.add_argument('--dry-run', action='store_true')
    args = parser.parse_args(replace_args_from_env())

    Runner(args).run()


if __name__ == '__main__':
    main()
