# coding=utf-8
from __future__ import unicode_literals

import argparse
from datetime import datetime
from datetime import timedelta
import json
import time
import webbrowser

from dateutil import parser as date_parser
import os
from prompt_toolkit import prompt
from prompt_toolkit.completion import Completer, Completion
from prompt_toolkit.validation import Validator, ValidationError
import requests
from travel.hotels.tools.cli import tvm
from travel.hotels.tools.cli import commands
from travel.hotels.tools.cli import client


class HotelsApiClient(object):
    def __init__(self, host, port, ticket):
        self.host = host
        self.port = port
        self.ticket = ticket

    def get_url(self, method):
        return "http://{}:{}/api/{}".format(self.host, self.port, method)

    def suggest(self, query):
        try:
            res = self.get("suggest/v1", query=query)
            return json.loads(res)
        except:
            return []

    def search_serp(self, search_object, checkin, checkout, num_adults, children):
        searchMethodName = search_object['request_template']['method_name']
        params = search_object['request_template']['params']
        params["checkin_date"] = checkin
        params["checkout_date"] = checkout
        params["num_adults"] = num_adults
        params["children_ages"] = ",".join(children)
        json_resp = json.loads(self.get(searchMethodName, **params))
        return json_resp['url'], json_resp["text_template"], json_resp['redirect_timeout']

    def get(self, method, **kwargs):
        url = self.get_url(method)
        headers = {'X-Ya-Service-Ticket': self.ticket}
        resp = requests.get(url, params=kwargs, headers=headers)
        if resp.status_code == 200:
            return resp.content
        else:
            print(resp.content)
            resp.raise_for_status()


completions_cache = {}


def get_completions(client, text):
    resp_list = client.suggest(text)
    completions_cache.clear()
    for resp in resp_list:
        repeats = 1
        name = resp['name']
        while True:
            if name in completions_cache:
                repeats += 1
                name = "{} ({})".format(completions_cache[name]['name'], repeats)
            else:
                break
        completions_cache[name] = resp
        descr = u"{}: {}".format(resp['type_name'], resp["description"])
        yield name, descr


class HotelCompleter(Completer):
    def __init__(self, client, completion_func):
        self.client = client
        self.completion_func = completion_func

    def get_completions(self, document, complete_event):
        for name, description in self.completion_func(self.client, document.text):
            yield Completion(name, -1 * len(document.text), display_meta=description)


class SuggestValidator(Validator):
    def __init__(self, client):
        self.client = client

    def validate(self, document):
        if document.text not in completions_cache:
            raise ValidationError(len(document.text), message="No match")


class IntValidator(Validator):
    def __init__(self, min_value, max_value):
        self.min_value = min_value
        self.max_value = max_value

    def validate(self, document):
        try:
            val = int(document.text)
        except:
            raise ValidationError(len(document.text), message="Invalid number")

        if val > self.max_value:
            raise ValidationError(len(document.text), message="Value too large")
        if val < self.min_value:
            raise ValidationError(len(document.text), message="Value too low")


class DateValidator(Validator):
    def validate(self, document):
        try:
            date = date_parser.parse(document.text).date()
        except Exception as e:
            raise ValidationError(len(document.text), message="Incorrect date: {}".format(e.message))
        if date < datetime.now().date():
            raise ValidationError(len(document.text), message="Past date")
            # if date > datetime.now().date() + timedelta(days=100):
            #     raise ValidationError(len(document.text), message="Too far in future")


def main():
    parser = argparse.ArgumentParser()
    # parser.add_argument("--host", default="localhost")
    parser.add_argument("--host", default="api.travel-balancer-test.yandex.net")
    parser.add_argument("--port", default=80, type=int)
    # parser.add_argument("--port", default=8081, type=int)
    parser.add_argument("--cli-tvm-service-id", default=2002574, type=int)
    parser.add_argument("--api-tvm-service-id", default=2002548, type=int)
    parser.add_argument("--cli-tvm-secret")
    parser.add_argument("--cli-service-ticket", help="Use provided service ticket (For those who have only TVM ssh user role this can be obtained via tvmknife utility)")
    parser.add_argument("--skip-authentication", action='store_true', default=False)
    subparsers = parser.add_subparsers()
    command_list = [command_class(subparsers) for command_class in commands.ALL]
    args = parser.parse_args()
    if args.skip_authentication:
        ticket = None
    elif args.cli_service_ticket:
        ticket = args.cli_service_ticket
    else:
        if args.cli_tvm_secret is None:
            if 'TVM_SECRET' not in os.environ:
                print("TVM-секрет должен быть указан или параметром `--cli-tvm-secret` или переменной окружения "
                      "`TVM_SECRET`")
                return
            secret = os.environ['TVM_SECRET']
        else:
            secret = args.cli_tvm_secret
        tvm_client = tvm.make_tvm_client(args.cli_tvm_service_id, secret, {'api': args.api_tvm_service_id})
        ticket = tvm_client.get_service_ticket_for("api")
    hotels_client = client.HotelsApiClient(args.host, args.port, ticket)
    for command in command_list:
        command.set_client(hotels_client)

    args.func(args)


def old_main():
    parser.add_argument("--host", default="api.travel-balancer-test.yandex.net")
    parser.add_argument("--port", default=80, type=int)
    parser.add_argument("--suggest", default="")
    parser.add_argument("--skip-authentication", action='store_true', default=False)
    parser.add_argument("--cli-tvm-service-id", default=2002574, type=int)
    parser.add_argument("--api-tvm-service-id", default=2002548, type=int)
    parser.add_argument("--cli-tvm-secret")
    args = parser.parse_args()
    if args.skip_authentication:
        ticket = None
    else:
        if args.cli_tvm_secret is None:
            secret = os.environ['TVM_SECRET']
        else:
            secret = args.tvm_secret
        tvm_client = tvm.make_tvm_client(args.cli_tvm_service_id, secret, {'api': args.api_tvm_service_id})
        ticket = tvm_client.get_service_ticket_for("api")

    client = HotelsApiClient(args.host, args.port, ticket)
    completer = HotelCompleter(client, get_completions)
    suggest_validator = SuggestValidator(client)

    if args.suggest:
        for name, description in get_completions(client, args.suggest):
            print u"{} ({})".format(name, description)
    else:
        try:
            text = prompt('(Отель или регион): ', completer=completer, validator=suggest_validator)
            completion = completions_cache[text]
            today = datetime.now().date()
            tomorrow = datetime.now().date() + timedelta(days=1)
            checkin = prompt('(Дата заезда): ', default=unicode(today.strftime("%Y-%m-%d")),
                             validator=DateValidator())
            checkout = prompt('(Дата выезда): ', default=unicode(tomorrow.strftime("%Y-%m-%d")),
                              validator=DateValidator())
            num_adults = int(prompt('(Колчество взрослых): ', default='2', validator=IntValidator(1, 8)))
            num_children = int(prompt('(Колчество детей): ', default='0', validator=IntValidator(0, 8)))
            children = []
            for i in xrange(1, num_children + 1):
                children.append(prompt('(Возраст ребенка {}): '.format(i)))

            res, template, timeout = client.search_serp(completion, checkin, checkout, num_adults, children)
            if template['type'] == 'REGION':
                msg_template = u"Вы ищете отель в {name} c {dateFrom} по {dateTo} для {adults}{children}."
            else:
                msg_template = u"Вы ищете отель «{name}» c {dateFrom} по {dateTo} для {adults}{children}."
            print(msg_template.format(name=template['name'],
                                      dateFrom=template['date_from'],
                                      dateTo=template['date_to'],
                                      adults=template['adults'],
                                      children=" и {}".format(template["children"])
                                      if template.get("children") else ""))
            print(u"Теперь отели можно искать прямо на странице поиска в Яндексе.")
            print(u"Переход на страницу произойдет автоматически через {} секунд.".format(timeout / 1000.0))
            time.sleep(3)
            webbrowser.open(res)
        except KeyboardInterrupt:
            print "Bye!"


if __name__ == '__main__':
    main()
