# coding=utf-8
from __future__ import unicode_literals

import base
import datetime
import collections
import time
from datetime import timedelta
from dateutil import parser as date_parser
from urlparse import urlparse, parse_qs
import webbrowser

from prompt_toolkit import prompt
from prompt_toolkit.completion import Completer, Completion
from prompt_toolkit.validation import Validator, ValidationError
import requests

from travel.hotels.tools.cli import tableprinter


class BookingFlowCommand(base.Command):
    name = "booking-flow"

    def add_arguments(self):
        self.sub_parser.add_argument("--operator", action="append")
        self.sub_parser.add_argument("--order-id")

    def handle(self, args):
        order_id = None
        try:
            details_printed = False
            payment_url_printed = False
            payment_started = False

            if args.order_id is None:
                operator, url = self.search_offer(args.operator)
                if url is None:
                    print("Результатов не найдено :(")
                    return
                if operator != "Яндекс.Путешествия":
                    print("Продолжите бронирование на сайте партнера...")
                    time.sleep(1)
                    webbrowser.open(url)
                    print(url)
                    return
                label, token = self.get_label_and_token(url)
                if token is None:
                    print("Результатов не найдено :(")
                    return
                order_id = self.create_order(label, token)
                if order_id is None:
                    return
                details_printed = True
            else:
                order_id = args.order_id
            while True:
                order = self.client.get_order(order_id)
                if order is None:
                    print("Заказ не найден")
                    return
                if not details_printed:
                    self.print_order_info(order["order_info"], order["order_info"]["bed_groups"][0], order_id,
                                          order["status"], order["status"] != "AWAITS_PAYMENT",
                                          order["guest_info"]["guests"],
                                          order["confirmation_info"], order["payment"], order.get("error_info"))
                    details_printed = True
                    if order["status"] == "CONFIRMED":
                        return
                if order["status"] == "IN_PROGRESS":
                    time.sleep(1)
                    continue
                elif order["status"] == "RESERVED":
                    if not payment_started:
                        self.client.start_payment(order_id, "https://travel-prod.yandex.ru/hotels/")
                        payment_started = True
                    continue
                elif order["status"] == "AWAITS_PAYMENT":
                    if not payment_url_printed:
                        print("Нужно оплатить заказ...")
                        url = order["payment"]["payment_url"]
                        webbrowser.open(url)
                        print(url)
                        payment_url_printed = True
                    time.sleep(1)
                    continue
                elif order["status"] == "CONFIRMED":
                    self.print_order_info(order["order_info"], order["order_info"]["bed_groups"][0], order_id,
                                          order["status"], True, order["guest_info"]["guests"],
                                          order["confirmation_info"], order["payment"])
                    return
                else:
                    status = order["status"]
                    if order.get("error_info"):
                        status = "{} : {} ({})".format(status, order["error_info"]["type"],
                                                       order["error_info"]["message"])
                    print(status)
                    break
        except KeyboardInterrupt:
            if order_id is None:
                return
            else:
                print("\nЧтобы продолжить работу с заказом, выполните "
                      "`./hotels-cli booking-flow --order-id {}`".format(order_id))

    def create_order(self, label, token):
        order_info = self.client.get_offer_by_token(label, token)
        if len(order_info["bed_groups"]) > 1:
            bg_completer = ListCompleterValidator(order_info["bed_groups"],
                                                  lambda (x): (x["description"], x["description"], ""))
            bg_name = prompt("(Размещение): ", completer=bg_completer, validator=bg_completer)
            bed_group = bg_completer.get(bg_name)
        else:
            bed_group = order_info["bed_groups"][0]
        self.print_order_info(order_info, bed_group, None)
        if prompt("(Забронировать?) ", default="Да").lower() != "да":
            return None
        guests = []
        for i in xrange(order_info["request_info"]["num_adults"]):
            first_name = prompt("(Имя гостя " + str(i + 1) + "): ")
            last_name = prompt("(Фамилия гостя " + str(i + 1) + "): ")
            guests.append(dict(first_name=first_name, last_name=last_name))
        for age in order_info["request_info"]["child_ages"]:
            first_name = prompt("(Имя ребенка " + str(age) + " лет): ")
            last_name = prompt("(Фамилия ребенка " + str(age) + " лет): ")
            guests.append(dict(first_name=first_name, last_name=last_name))
        email = prompt("(Ваш email): ")
        phone = prompt("(Ваш телефон): ")
        order_id = self.client.create_order(order_info["travel_token"], order_info["label"], order_info["checksum"],
                                            "login", "uid", "", order_info["session_key"], bed_group["id"], email,
                                            phone, True, guests)
        print("Создан заказ {}".format(order_id))
        return order_id

    def search_offer(self, operators, fast=False):
        completion_cache = {}
        completer = SuggestCompleter(self.client, completion_cache)
        suggest_validator = SuggestValidator(completion_cache)
        text = prompt(u'(Отель или регион): ', completer=completer, validator=suggest_validator)
        search = completion_cache[text]
        # today = datetime.datetime.now().date() + timedelta(days=30)
        checkin = prompt('(Дата заезда): ', validator=DateValidator())
        if checkin == "":
            checkin = None
            checkout = None
            num_adults = None
            num_children = None
            children = None
        else:
            tomorrow = datetime.datetime.strptime(checkin, "%Y-%m-%d").date() + timedelta(days=1)
            checkout = prompt('(Дата выезда): ', default=unicode(tomorrow.strftime("%Y-%m-%d")),
                              validator=DateValidator())
            num_adults = int(prompt('(Колчество взрослых): ', default='2', validator=IntValidator(1, 8)))
            num_children = int(prompt('(Колчество детей): ', default='0', validator=IntValidator(0, 8)))
            children = []
            for i in xrange(1, num_children + 1):
                children.append(prompt('(Возраст ребенка {}): '.format(i)))
        if search["type_id"] == "hotel":
            permalink = search["request_template"]["params"]["permalink"]
            print("Ищем отель...")
            hotels_list = [r for r in
                           self.client.search_hotel_on_portal(permalink, checkin, checkout, num_adults, children,
                                                              operators)[
                               "hotels"] if
                           r["offers"]]
            if not hotels_list:
                return None, None
            assert len(hotels_list) == 1
        else:
            region_name = search["request_template"]["params"]["location_name"]
            print("Ищем отели...")
            hotels_list = [r for r in
                           self.client.search_region_on_portal(region_name, checkin, checkout, num_adults, children,
                                                               operators,
                                                               limit=30)["hotels"] if r["offers"]]

        if not hotels_list:
            return None, None
        if len(hotels_list) == 1:
            hotel = hotels_list[0]
            print("(Отель): " + hotel["name"])
        else:
            hotels_to_print = hotels_list[0:10]
            tableprinter.print_table(hotels_to_print,
                                     [("permalink", "id"), ("name", u"Название"), ("address", u"Адрес"),
                                      (["min_price", "price"], u"Цена от, руб")])
            if len(hotels_list) > len(hotels_to_print):
                print("...и еще {} отелей".format(len(hotels_list) - len(hotels_to_print)))
            hotel_completer = ListCompleterValidator(hotels_list, self.map_hotel)
            hotel_name = prompt("(Отель): ", completer=hotel_completer, validator=hotel_completer)
            hotel = hotel_completer.get(hotel_name)

        if len(hotel.get("offers", [])) == 0:
            return None, None

        operators_map = {}
        for offer in hotel["offers"]:
            operator = offer["operator_name"]
            if operator not in operators_map:
                operators_map[operator] = []
            operators_map[operator].append(offer)

        offers_to_print = hotel["offers"][0:10]
        tableprinter.print_table(offers_to_print,
                                 [("room_type", u"Номер"), ("operator_name", u"Оператор"), ("pansion", u"Еда"),
                                  ("free_cancellation", u"Отменяемый"), ("price", u"Цена, руб")])

        if len(hotel["offers"]) > 10:
            print("...и еще {} предложений от {} операторов ".format(len(hotel["offers"]) - 10, len(operators_map)))

        if len(operators_map) == 1:
            print("(Оператор): " + operators_map.keys()[0])
            operator_name = operators_map.keys()[0]
        else:
            operators_completer = ListCompleterValidator(operators_map.keys(), lambda (x): (x, x, ""))
            operator_name = prompt("(Оператор): ", completer=operators_completer, validator=operators_completer)
            tableprinter.print_table(operators_map[operator_name],
                                     [("room_type", u"Номер"), ("operator_name", u"Оператор"), ("pansion", u"Еда"),
                                      ("free_cancellation", u"Отменяемый"), ("price", u"Цена, руб")])
        offers = operators_map[operator_name]
        if len(offers) == 0:
            return None, None
        if len(offers) == 1:
            print("(Номер): " + offers[0]["room_type"])
            offer = offers[0]
        else:
            offer_completer = ListCompleterValidator(offers, self.map_offer)
            offer_name = prompt("(Номер): ", completer=offer_completer, validator=offer_completer)
            offer = offer_completer.get(offer_name)

        return operator_name, offer["link"]

    def print_order_info(self, order_info, bed_group, order_id=None, status=None, paid=False, guests=None,
                         confirmation_info=None,
                         payment_info=None, error_info=None):
        info = collections.OrderedDict()
        if order_id is not None:
            info["id"] = order_id
        if status or error_info:
            info["Состояние"] = status
            if error_info:
                info["Тип ошибки"] = error_info["type"]
                info["Сообщение"] = error_info["message"]
            info["_hid0"] = ""
        info["Отель"] = order_info["basic_hotel_info"]["name"]
        info["Адрес"] = order_info["basic_hotel_info"]["address"]
        info["Рейтинг"] = order_info["basic_hotel_info"]["rating"]
        info["_hid1"] = ""
        info["Заезд"] = order_info["request_info"]["checkin_date"]
        info["Выезд"] = order_info["request_info"]["checkout_date"]
        info["Гостей"] = order_info["request_info"]["num_adults"] + len(order_info["request_info"]["child_ages"])
        info["_hid2"] = ""
        info["Номер"] = order_info["partner_room_info"]["name"]
        info["Размещение"] = bed_group["description"]
        pansion_info = order_info.get("pansion_info")
        if pansion_info:
            info["Пансион"] = pansion_info["name"]
        info["_hid3"] = ""
        info["Цена за номер"] = order_info["rate_info"]["hotel_charges"]["totals"]["base"]["amount"] + ' ' + \
                                order_info["rate_info"]["hotel_charges"]["totals"]["base"]["currency"]
        for fee in order_info["rate_info"]["hotel_charges"]["totals"]["taxes_and_fees"]:
            info["Сбор " + fee["type"]] = fee["amount"] + ' ' + fee["currency"]
        if paid:
            totals_key = "Всего оплачено"
        else:
            totals_key = "Всего"
        info[totals_key] = order_info["rate_info"]["hotel_charges"]["totals"]["grand"]["amount"] + ' ' + \
                                order_info["rate_info"]["hotel_charges"]["totals"]["grand"]["currency"]
        if order_info["rate_info"]["extra_charges"]:
            for extra in order_info["rate_info"]["extra_charges"]:
                info["Сбор в отеле " + extra["type"]] = extra["payable"]["amount"] + ' ' + extra["payable"]["currency"]
        info["_hid4"] = ""
        if guests:
            i = 1
            for guest in guests:
                info["Гость " + str(i)] = guest["first_name"] + ' ' + guest["last_name"]
                i += 1
            info["_hid5"] = ""
        if not paid:
            info["К оплате сейчас"] = order_info["rate_info"]["hotel_charges"]["totals"]["grand"]["amount"] + ' ' + \
                                      order_info["rate_info"]["hotel_charges"]["totals"]["grand"]["currency"]
            info["_hid6"] = ""
        info["Возвратный тариф"] = "Да" if order_info["cancellation_info"]["refundable"] else "Нет"
        if order_info["cancellation_info"]["refundable"]:
            for penalty in order_info["cancellation_info"]["penalties"]:
                if penalty["type"] == "NO_PENALTY":
                    info["Без штрафов при возврате до"] = penalty["ends_at"]
                if penalty["type"] == "SOME_PENALTY":
                    penalty_name = "c " + penalty["starts_at"] + " по " + penalty["ends_at"]
                    info[penalty_name] = "Штраф " + penalty["amount"] + ' ' + penalty["currency"]
                if penalty["type"] == "FULL_PRICE":
                    info["Невозвратно после"] = penalty["starts_at"]
        if confirmation_info is not None:
            info["_hid7"] = ""
            info["_hid8"] = ""
            info["Код подтверждения партнера"] = confirmation_info["partner_confirmation_id"]
            info["Код подтверждения отеля"] = confirmation_info["hotel_confirmation_id"]
        if payment_info and payment_info["receipts"]:
            url = payment_info["receipts"][0]["url"]
            if url.endswith("?mode=mobile"):
                url = url[0:-len("?mode=mobile")]
            info["Чек"] = url

        tableprinter.print_table(info)

    def get_label_and_token(self, link):
        resp = requests.get(link, allow_redirects=False)
        assert resp.status_code == 302
        redir_url = resp.headers['Location']
        parsed = parse_qs(urlparse(redir_url).query)
        return parsed["label"][0], parsed["token"][0]

    def map_offer(self, offer):
        return offer["room_type"], offer["room_type"], "{}, {} руб".format(offer["pansion"], offer["price"])

    def map_hotel(self, hotel):
        return hotel["name"] + hotel["address"], hotel["name"], hotel["address"]


class ListCompleterValidator(Completer, Validator):
    def validate(self, document):
        if document.text not in self.name_map:
            raise ValidationError(len(document.text), message="No match")

    def __init__(self, value_list, mapper):
        self.value_list = value_list
        self.mapper = mapper
        self.name_map = {}
        for value in value_list:
            _, name, _ = mapper(value)
            self.name_map[name] = value

    def get_completions(self, document, complete_event):
        for item in self.value_list:
            key, name, description = self.mapper(item)
            if document.text.lower() in key.lower():
                yield Completion(name, -1 * len(document.text), display_meta=description)

    def get(self, name):
        return self.name_map[name]


class SuggestCompleter(Completer):
    def __init__(self, client, completion_cache):
        self.client = client
        self.completion_cache = completion_cache

    def get_completions(self, document, complete_event):
        for name, description in self.complete(document.text):
            yield Completion(name, -1 * len(document.text), display_meta=description)

    def complete(self, text):
        resp_list = self.client.suggest(text)
        self.completion_cache.clear()
        for resp in resp_list:
            repeats = 1
            name = resp['name']
            while True:
                if name in self.completion_cache:
                    repeats += 1
                    name = "{} ({})".format(self.completion_cache[name]['name'], repeats)
                else:
                    break
            self.completion_cache[name] = resp
            descr = "{}: {}".format(resp['type_name'], resp["description"])
            yield name, descr


class SuggestValidator(Validator):
    def __init__(self, completion_cache):
        self.completion_cache = completion_cache

    def validate(self, document):
        if document.text not in self.completion_cache:
            raise ValidationError(len(document.text), message="No match")


class DateValidator(Validator):
    def validate(self, document):
        if not document.text:
            return
        try:
            date = date_parser.parse(document.text).date()
        except Exception as e:
            raise ValidationError(len(document.text), message="Incorrect date: {}".format(e.message))
        if date < datetime.datetime.now().date():
            raise ValidationError(len(document.text), message="Past date")


class IntValidator(Validator):
    def __init__(self, min_value, max_value):
        self.min_value = min_value
        self.max_value = max_value

    def validate(self, document):
        try:
            val = int(document.text)
        except:
            raise ValidationError(len(document.text), message="Invalid number")

        if val > self.max_value:
            raise ValidationError(len(document.text), message="Value too large")
        if val < self.min_value:
            raise ValidationError(len(document.text), message="Value too low")
