# -*- coding: utf-8 -*-
import json
import datetime
from dataclasses import dataclass
from enum import Enum

from typing import List, Optional, Dict, Union, Any

from yt.wrapper import YtClient, ypath_split

from travel.hotels.lib.python3.yt import ytlib
from travel.hotels.tools.dataset_curator.type_schemas import BaseTypeSchema


ENV_SUFFIXES = ['_prod', '_testing']


class VersioningScheme(Enum):
    SINGLE = 'single'
    PERIODIC = 'periodic'
    PERIODIC_WITH_LATEST = 'periodic_with_latest'
    LOGS = 'logs'


class DatasetType(Enum):
    SINGLE_TABLE = 'single_table'
    DIRECTORY = 'directory'


class CleanupAgeDetectionMode(Enum):
    BY_NAME = 'by_name'


@dataclass
class ColumnConfiguration:
    type: BaseTypeSchema
    description: str


@dataclass
class UniquenessValidationConfiguration:
    id: str
    key: List[str]


@dataclass
class PerRowValidationConfiguration:
    id: str
    is_valid_expr: str


@dataclass
class PerTableValidationConfiguration:
    id: str
    init_expr: str
    step_expr: str
    result_expr: str


@dataclass
class TableConfiguration:
    allow_unknown_columns: bool
    columns: Optional[Dict[str, ColumnConfiguration]]
    subpath: str
    ignored: bool
    uniqueness_validations: List[UniquenessValidationConfiguration]
    per_row_validations: List[PerRowValidationConfiguration]
    per_table_validations: List[PerTableValidationConfiguration]

    def get_id(self):
        return self.subpath or '<single>'


class BuilderType(Enum):
    YQL = 'yql'


@dataclass
class YqlBuilderConfiguration:
    query_path: str
    yql_params: dict


@dataclass
class BuilderConfiguration:
    builder_type: BuilderType
    versioned_process: bool
    create_latest: bool
    transfer_results: bool
    skip_empty: bool
    skip_unexistent: bool
    specific_builder_configuration: Union[YqlBuilderConfiguration]


@dataclass
class SbPlannerConfiguration:
    sb_planner_args: Dict[str, Any]
    builder_args_per_suffix: Dict[str, Dict[str, str]]


@dataclass
class BuildConfiguration:
    builder: Optional[BuilderConfiguration]
    sb_planner: Optional[SbPlannerConfiguration]


@dataclass
class CleanupConfiguration:
    mode: CleanupAgeDetectionMode
    yt_clusters: Optional[List[str]]
    subpaths: Optional[List[str]]
    keep_newer_than: Optional[datetime.timedelta]
    keep_last: int


class DatasetVersion:
    def __init__(self, dataset: 'Dataset', path: str):
        self.dataset = dataset
        dataset_dir, name = ypath_split(path)
        if dataset_dir != dataset.yt_path:
            raise Exception(f'Expected DatasetVersion\'s path ({path}) to be a child of Dataset\'s path ({dataset.yt_path})')
        self.node_name = name
        self.path = path

    def __str__(self):
        return json.dumps(self)

    def __repr__(self):
        return self.__str__()

    def get_dataset_id(self):
        return self.dataset.name

    def get_id(self):
        return 'ver:' + self.node_name


@dataclass
class Dataset:
    name: str
    owner: str
    yt_path: str
    yt_clusters: List[str]
    versioning_scheme: VersioningScheme
    max_age: Optional[datetime.timedelta]
    dataset_type: DatasetType
    table: Optional[TableConfiguration]
    tables: Optional[List[TableConfiguration]]
    build: Optional[BuildConfiguration]
    template_vars: Dict[str, Any]
    cleanup_rules: List[CleanupConfiguration]

    def get_dataset_versions(self, yt_client: YtClient, ignore_max_age=False) -> List[DatasetVersion]:
        def parse_time_attr(attr) -> datetime.datetime:
            return datetime.datetime.strptime(attr[:-8], '%Y-%m-%dT%H:%M:%S')

        def list_versions(path):
            attr = 'creation_time'
            res = []
            for node in yt_client.list(path, attributes=[attr], absolute=True):
                if ignore_max_age or self.max_age is None or (datetime.datetime.utcnow() - parse_time_attr(node.attributes[attr]) < self.max_age):
                    res.append(DatasetVersion(self, str(node)))
            return res

        if self.versioning_scheme == VersioningScheme.SINGLE:
            if not yt_client.exists(self.yt_path):
                return []
            return [DatasetVersion(self, self.yt_path)]
        elif self.versioning_scheme == VersioningScheme.PERIODIC:
            return list_versions(self.yt_path)
        elif self.versioning_scheme == VersioningScheme.PERIODIC_WITH_LATEST:
            return [x for x in list_versions(self.yt_path) if not x.path.endswith('/latest')]
        elif self.versioning_scheme == VersioningScheme.LOGS:
            return list_versions(ytlib.join(self.yt_path, '1d')) + list_versions(ytlib.join(self.yt_path, '1h'))
        else:
            raise Exception(f'Unknown VersioningScheme: {self.versioning_scheme}')
