from dataclasses import dataclass
from typing import Dict, Any, Callable, List

from travel.hotels.lib.python3.yql import yqllib
from library.python import resource
from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient


@dataclass
class BuildContext:
    _yql_client: YqlClient
    _inner_build_context: 'InnerBuildContext'
    yt_client: YtClient
    general_dataset_path: str
    current_dataset_path: str
    current_dataset_version_path: str
    _all_async_yql_reqs: List

    def run_yql_builder(self, yql_resource_name: str, yql_params: Dict[str, Any], sync=True):
        attaches = {}
        for lib_name, lib_data in resource.iteritems():
            if lib_name.endswith('lib.sql'):
                content = lib_data.decode('utf-8')
                prefix = 'USE `{}`;\n'.format(self._yql_client.config.db)
                content = prefix + content
                attaches[lib_name.replace('/', '_')] = content
            if lib_name.endswith('.py'):
                prefix = '/'.join(yql_resource_name.split('/')[:-1])
                if lib_name.startswith(prefix + '/'):
                    content = lib_data.decode('utf-8')
                    attaches[lib_name.split('/')[-1]] = content

        req = yqllib.run_yql_file(
            self._yql_client,
            resource_name=yql_resource_name,
            project_name='Travel',
            title=f'Dataset curator: Build {self._inner_build_context.name}',
            parameters=yql_params,
            attaches=attaches,
            sync=sync
        )
        if not sync:
            self._all_async_yql_reqs.append(req)
        return req

    def wait_all_yql_reqs(self):
        yqllib.wait_results(*self._all_async_yql_reqs)
        self._all_async_yql_reqs = []

    def _finish(self):
        self.wait_all_yql_reqs()


@dataclass
class InnerBuildContext:
    name: str
    func: Callable[[BuildContext, Any], None]
    yt_cluster: str
    versioned_path: bool
    create_latest: bool
    transfer_results: bool
    skip_empty: bool
    skip_unexistent: bool


class BuildersRegistry:
    builders: Dict[str, InnerBuildContext] = {}

    def new_builder(self, name, yt_cluster='hahn', versioned_process=False, create_latest=False, transfer_results=False, skip_empty=False, skip_unexistent=False):
        def decorator(func: Callable[[BuildContext, Any], None]):
            self.builders[name] = InnerBuildContext(name, func, yt_cluster, versioned_process, create_latest, transfer_results, skip_empty, skip_unexistent)

        return decorator
