$get_dist = Python3::get_dist(@@
from shapely.wkb import loads
from shapely.geometry import Point
from shapely.ops import nearest_points
import pyproj
from yql.typing import *

def get_dist(data: String, lat: Double, lon: Double) -> Double:
    a, b = nearest_points(loads(data.decode(), 'hex'), Point(lon, lat))
    geod = pyproj.Geod(ellps='WGS84')
    angle1, angle2, distance = geod.inv(a.x, a.y, b.x, b.y)
    return distance
@@);

$get_dist_multi = Python3::get_dist_multi(@@
from shapely.wkb import loads
from shapely.geometry import Point
from shapely.ops import nearest_points
import pyproj
from yql.typing import *


def _get_dist(geod, geom, point):
    a, b = nearest_points(geom, point)
    angle1, angle2, distance = geod.inv(a.x, a.y, b.x, b.y)
    return distance


def get_dist_multi(data: String, points: List[Struct['Lat': Double, 'Lon': Double]], to_exterior: Bool) -> List[Double]:
    geom = loads(data.decode(), 'hex')
    geod = pyproj.Geod(ellps='WGS84')

    res = []

    for point in points:
        p = Point(point.Lon, point.Lat)
        if to_exterior and geom.type == 'MultiPolygon':
            res.append(min([_get_dist(geod, poly.exterior, p) for poly in list(geom)]))
        elif to_exterior and geom.type == 'Polygon':
            res.append(_get_dist(geod, geom.exterior, p))
        else:
            res.append(_get_dist(geod, geom, p))

    return res
@@);

$get_dist_points = Python3::get_dist(@@
import pyproj
from yql.typing import *

def get_dist(a_lat: Double, a_lon: Double, b_lat: Double, b_lon: Double) -> Double:
    geod = pyproj.Geod(ellps='WGS84')
    angle1, angle2, distance = geod.inv(a_lon, a_lat, b_lon, b_lat)
    return distance
@@);

$get_center = Python3::get_center(@@
from shapely.wkb import loads
from yql.typing import *

def get_center(data: String) -> Tuple[Double, Double]:
    center = loads(data.decode(), 'hex').centroid
    return center.x, center.y
@@);

$get_area = Python3::get_area(@@
from shapely.wkb import loads
from shapely.ops import transform
import pyproj
from yql.typing import *


def get_area(data: String) -> Double:
    geom = loads(data.decode(), 'hex')

    projection = pyproj.Proj(proj='aea', lat_1=geom.bounds[1], lat_2=geom.bounds[3])

    return transform(projection, geom).area
@@);

$get_bounds = Python3::get_bounds(@@
from shapely.wkb import loads
from yql.typing import *

def get_bounds(data: String) -> Tuple[Double, Double, Double, Double]:
    geom = loads(data.decode(), 'hex')
    if geom.type == 'MultiPolygon':
        points = []
        for poly in list(geom):
            points += [list(x) for x in list(poly.exterior.coords)]
    elif geom.type == 'MultiLineString':
        points = []
        for line in list(geom):
            points += [list(x) for x in list(line.coords)]
    elif geom.type == 'Point' or geom.type == 'LineString':
        points = [list(x) for x in list(geom.coords)]
    else:
        points = [list(x) for x in list(geom.exterior.coords)]

    lons = sorted([p[0] for p in points])
    pairs = zip(lons, lons[1:] + lons[:1])
    right, left = max([((p[1] - p[0] + 360) % 360, p) for p in pairs])[1]

    minx, miny, maxx, maxy = loads(data.decode(), 'hex').bounds

    return left, miny, right, maxy
@@);

DEFINE SUBQUERY $get_places($ft_type_ids) AS
    SELECT
        `ft_id`,
        `disp_class`,
        `disp_class_navi`,
        `disp_class_tweak`,
        `disp_class_tweak_navi`,
        `ft_type_id`,
        `icon_class`,
        `isocode`,
        `p_ft_id`,
        `rubric_id`,
        `search_class`,
        `subcode`
    FROM RANGE(`//home/maps/core/garden/stable/ymapsdf/latest`, ``, ``, `ft`)
    WHERE ListHas($ft_type_ids, `ft_type_id`)
END DEFINE;


DEFINE SUBQUERY $get_places_with_geom($ft_type_ids) AS
    SELECT
        places.`ft_id` AS ft_id,
        Unwrap(geom.`shape`) AS shape,
        $get_center(Unwrap(geom.`shape`)).0 AS center_lon,
        $get_center(Unwrap(geom.`shape`)).1 AS center_lat,
    FROM $get_places($ft_type_ids) AS places
    LEFT JOIN RANGE(`//home/maps/core/garden/stable/ymapsdf/latest`, ``, ``, `ft_geom`) AS geom USING (ft_id)
    WHERE geom.`shape` IS NOT NULL
END DEFINE;

DEFINE SUBQUERY $get_places_with_area_filter($ft_type_ids, $min_area) AS
    $places_with_geom = (SELECT * FROM $get_places_with_geom($ft_type_ids));

    SELECT *
    FROM $places_with_geom
    WHERE $get_area(`shape`) >= $min_area

END DEFINE;

DEFINE SUBQUERY $get_min_dists_inner($ft_type_ids, $min_area, $pre_filter_threshold, $to_exterior) AS
    DEFINE SUBQUERY $get_simple() AS
        SELECT * FROM $get_places_with_geom($ft_type_ids)
    END DEFINE;

    DEFINE SUBQUERY $get_with_area_filter() AS
        SELECT * FROM $get_places_with_area_filter($ft_type_ids, $min_area)
    END DEFINE;

    $places_data = ($world) -> {
        $q = EvaluateCode(
            IF(
                $min_area IS NULL,
                QuoteCode($get_simple),
                QuoteCode($get_with_area_filter)
            )
        );
        return $q($world)
    };

    $places_with_geom = (
        SELECT
            `center_lon`,
            `center_lat`,
            `shape`,
            `ft_id`,
            Geo::CalculatePointsDifference($get_bounds(`shape`).1, $get_bounds(`shape`).0, $get_bounds(`shape`).3, $get_bounds(`shape`).2) AS diagonal_length,
        FROM $places_data()
    );

    $batches_count = 1000;

    $hotels = (
        SELECT
            Digest::NumericHash(Unwrap(CAST(altay.permalink AS UInt64))) % $batches_count AS batch,
            <|
                Permalink: altay.permalink,
                Lat: Unwrap(exported_company.Geo.Location.Pos.Lat),
                Lon: Unwrap(exported_company.Geo.Location.Pos.Lon),
            |> AS hotel_data,
        FROM `//home/altay/db/export/current-state/exported/company` AS altay
        JOIN `//home/travel/prod/general/altay_mappings/latest/hotels_permalinks_published` AS hotels_published ON (altay.permalink == hotels_published.permalink)
        WHERE exported_company.Geo.Location.Pos.Lat IS NOT NULL AND exported_company.Geo.Location.Pos.Lon IS NOT NULL
    );

    $hotel_batches = (
        SELECT
            AGGREGATE_LIST(`hotel_data`) AS hotel_datas,
        FROM $hotels
        GROUP BY `batch`
    );

    $hotels_with_places_pairs = (
        SELECT
            hotel_batches.hotel_datas AS hotel_datas,
            places.center_lon AS place_center_lon,
            places.center_lat AS place_center_lat,
            places.shape AS place_shape,
            places.ft_id AS place_ft_id,
            places.diagonal_length AS place_diagonal_length,
        FROM $hotel_batches AS hotel_batches
        CROSS JOIN $places_with_geom AS places
    );

    $hotels_with_places_pairs_dists_batches = (
        SELECT
            (($hotel_datas, $place_center_lat, $place_center_lon, $place_shape, $place_diagonal_length) -> {
                $filtered = IF($pre_filter_threshold IS NULL, $hotel_datas, ListFilter($hotel_datas, ($data) -> (
                    Geo::CalculatePointsDifference($data.Lat, $data.Lon, $place_center_lat, $place_center_lon) < $pre_filter_threshold + $place_diagonal_length + 10000
                )));
                $points = ListMap($filtered, ($data) -> {
                    RETURN <|
                        Lat: $data.Lat,
                        Lon: $data.Lon,
                    |>;
                });
                $dists = $get_dist_multi($place_shape, $points, $to_exterior);
                RETURN ListMap(ListZip($filtered, $dists), ($pair) -> (<|
                    Permalink: ($pair.0).Permalink,
                    Dist: $pair.1,
                |>));
            })(`hotel_datas`, `place_center_lat`, `place_center_lon`, `place_shape`, `place_diagonal_length`) AS dists,
            place_ft_id AS place_ft_id,
        FROM $hotels_with_places_pairs
    );

    $hotels_with_places_pairs_true_dists = (
        SELECT
            dists.Permalink AS permalink,
            dists.Dist AS dist,
            place_ft_id,
        FROM $hotels_with_places_pairs_dists_batches
        FLATTEN LIST BY dists
    );

    SELECT
        `permalink`,
        MIN(`dist`) AS dist,
        MIN_BY(`place_ft_id`, `dist`) AS place_ft_id,
    FROM $hotels_with_places_pairs_true_dists
    GROUP BY permalink
END DEFINE;

DEFINE SUBQUERY $get_min_dists($ft_type_ids, $to_exterior) AS
    SELECT *
    FROM $get_min_dists_inner($ft_type_ids, NULL, NULL, $to_exterior)
END DEFINE;

DEFINE SUBQUERY $get_min_dists_with_area_filter($ft_type_ids, $min_area, $to_exterior) AS
    SELECT *
    FROM $get_min_dists_inner($ft_type_ids, $min_area, NULL, $to_exterior)
END DEFINE;

DEFINE SUBQUERY $get_min_dists_with_area_filter_and_pre_filter($ft_type_ids, $min_area, $pre_filter_threshold, $to_exterior) AS
    SELECT *
    FROM $get_min_dists_inner($ft_type_ids, $min_area, $pre_filter_threshold, $to_exterior)
END DEFINE;

DEFINE SUBQUERY $get_places_with_centers($ft_type_ids) AS
    SELECT
        places.`ft_id` AS ft_id,
        nodes.`x` AS x,
        nodes.`y` AS y,
    FROM $get_places($ft_type_ids) AS places
    JOIN RANGE(`//home/maps/core/garden/stable/ymapsdf/latest`, ``, ``, `ft_center`) AS centers ON (places.ft_id == centers.ft_id)
    JOIN RANGE(`//home/maps/core/garden/stable/ymapsdf/latest`, ``, ``, `node`) AS nodes ON (centers.node_id ==  nodes.node_id)
END DEFINE;

DEFINE SUBQUERY $get_min_dists_to_centers($ft_type_ids) AS
    $places_with_centers = (SELECT * FROM $get_places_with_centers($ft_type_ids));

    $hotels = (
        SELECT
            altay.permalink AS permalink,
            Unwrap(exported_company.Geo.Location.Pos.Lat) AS lat,
            Unwrap(exported_company.Geo.Location.Pos.Lon) AS lon,
        FROM `//home/altay/db/export/current-state/exported/company` AS altay
        JOIN `//home/travel/prod/general/altay_mappings/latest/hotels_permalinks_published` AS hotels_published ON (altay.permalink == hotels_published.permalink)
        WHERE exported_company.Geo.Location.Pos.Lat IS NOT NULL AND exported_company.Geo.Location.Pos.Lon IS NOT NULL
    );

    $hotels_with_places_pairs = (
        SELECT
            permalink,
            lat,
            lon,
            places.x AS place_center_lon,
            places.y AS place_center_lat,
            places.ft_id AS place_ft_id,
        FROM $hotels AS hotels
        CROSS JOIN $places_with_centers AS places
    );

    $hotels_with_dists = (
        SELECT
            permalink,
            Geo::CalculatePointsDifference(lat, lon, place_center_lat, place_center_lon) AS dist,
            place_ft_id,
        FROM $hotels_with_places_pairs
    );

    SELECT
        `permalink`,
        MIN(`dist`) AS dist,
        MIN_BY(`place_ft_id`, `dist`) AS place_ft_id,
    FROM $hotels_with_dists
    GROUP BY permalink
END DEFINE;

EXPORT $get_dist, $get_dist_points, $get_center, $get_min_dists, $get_min_dists_with_area_filter, $get_min_dists_with_area_filter_and_pre_filter, $get_min_dists_to_centers;
