import datetime
import logging

from typing import Any, Dict
from yt.wrapper import ypath_split

from travel.hotels.lib.python3.yt import ytlib

from travel.hotels.tools.dataset_curator.dataset_build_configuration import BuildContext
from travel.hotels.tools.dataset_curator.datasets.registry import get_registry_for_build

LOG = logging.getLogger(__name__)

registry = get_registry_for_build()


@registry.new_builder('partner_id_original_id_observed_prices', transfer_results=True)
def partner_id_original_id_observed_prices_builder(ctx: BuildContext, args: Dict[str, Any]):
    logs_table_name = ypath_split(args['searcher_logs_path'])[1]

    ctx.run_yql_builder(
        'prices_and_availabilities/build_partner_id_original_id_observed_prices.yql',
        yql_params={
            '$input_path': args['searcher_logs_path'],
            '$output_path': str(ytlib.join(ctx.current_dataset_path, logs_table_name))
        }
    )


@registry.new_builder('partner_id_original_id_per_night_prices')
def partner_id_original_id_per_night_prices_builder(ctx: BuildContext, args: Dict[str, Any]):
    observed_prices_table_name = ypath_split(args['observed_prices_path'])[1]

    ctx.run_yql_builder(
        'prices_and_availabilities/build_partner_id_original_id_per_night_prices.yql',
        yql_params={
            '$observed_prices_path': args['observed_prices_path'],
            '$output_path': ytlib.join(ctx.current_dataset_path, observed_prices_table_name)
        }
    )


@registry.new_builder('aggregated_prices', create_latest=True, transfer_results=True)
def aggregated_prices_builder(ctx: BuildContext, args: Dict[str, Any]):
    date_format = '%Y-%m-%d'

    per_night_prices_dir, per_night_prices_table_name = ypath_split(args['per_night_prices_path'])
    date_parsed = datetime.datetime.strptime(per_night_prices_table_name, date_format)

    current_version_path = ytlib.join(ctx.current_dataset_path, per_night_prices_table_name)

    ctx.yt_client.create('map_node', current_version_path, recursive=True)
    cluster_permalink_per_night_prices_path = ytlib.join(current_version_path, 'cluster_permalink_per_night_prices')
    hotel_id_per_night_prices_path = ytlib.join(current_version_path, 'hotel_id_per_night_prices')
    num_tables_for_aggregation = int(args['num_tables_for_aggregation'])

    ctx.run_yql_builder(
        'prices_and_availabilities/build_cluster_permalink_per_night_prices.yql',
        yql_params={
            '$per_night_prices_dir': per_night_prices_dir,
            '$per_night_prices_from': datetime.datetime.strftime(date_parsed - datetime.timedelta(num_tables_for_aggregation - 1), date_format),
            '$per_night_prices_to': datetime.datetime.strftime(date_parsed, date_format),
            '$partners_config_path': args['partners_config_path'],
            '$output_path': cluster_permalink_per_night_prices_path,
            '$ema_ndays': num_tables_for_aggregation,
        }
    )

    ctx.run_yql_builder(
        'prices_and_availabilities/build_cluster_permalink_prices.yql',
        yql_params={
            '$cluster_permalink_per_night_prices_path': cluster_permalink_per_night_prices_path,
            '$output_path': ytlib.join(current_version_path, 'cluster_permalink_prices')
        }
    )

    if args.get('no_hotel_id_prices', False):
        # adhoc solution
        schema = ytlib.schema_from_dict({
            'Date': 'string',
            'Occupancy': 'string',
            'PartnerId': 'string',
            'OriginalId': 'string',
            'SingleNightMinPrice': 'int64',
            'SingleNightMaxPrice': 'int64',
            'UniformlyInterpolatedMinPrice': 'int64',
            'UniformlyInterpolatedMaxPrice': 'int64',
            'SingleNightOrUniformlyInterpolatedMinPrice': 'int64',
            'SingleNightOrUniformlyInterpolatedMaxPrice': 'int64',
            'InterpolatedAvailableCount_Background': 'double',
            'InterpolatedAvailableCount_Interactive': 'double',
            'SingleNightAvailableCount_Background': 'double',
            'SingleNightAvailableCount_Interactive': 'double',
            'InterpolatedUnavailableCount_Background': 'double',
            'InterpolatedUnavailableCount_Interactive': 'double',
            'SingleNightUnavailableCount_Background': 'double',
            'SingleNightUnavailableCount_Interactive': 'double',
            'LastAvailabilityProbability': 'double',
            'MovingAverageAvailabilityProbability': 'double',
        })
        ctx.yt_client.create('table', hotel_id_per_night_prices_path, recursive=True, attributes={"schema": schema})
    else:
        ctx.run_yql_builder(
            'prices_and_availabilities/build_aggr_hotel_id_per_night_prices.yql',
            yql_params={
                '$per_night_prices_dir': per_night_prices_dir,
                '$per_night_prices_from': datetime.datetime.strftime(date_parsed - datetime.timedelta(num_tables_for_aggregation - 1), date_format),
                '$per_night_prices_to': datetime.datetime.strftime(date_parsed, date_format),
                '$output_path': hotel_id_per_night_prices_path,
                '$ema_ndays': num_tables_for_aggregation,
            }
        )


@registry.new_builder('price_filter_data', versioned_process=True, create_latest=True, transfer_results=True)
def price_filter_data_builder(ctx: BuildContext, args: Dict[str, Any]):
    ctx.run_yql_builder(
        'prices_and_availabilities/build_price_filter_data.yql',
        yql_params={
            '$cluster_permalink_per_night_prices': args['cluster_permalink_per_night_prices'],
            '$cluster_permalink_prices': args['cluster_permalink_prices'],
            '$min_date': datetime.date.today().isoformat(),
            '$output_path': ytlib.join(ctx.current_dataset_version_path, 'price_filter_data'),
        }
    )


@registry.new_builder('aggregated_availabilities', create_latest=True, transfer_results=True)
def aggregated_availabilities_builder(ctx: BuildContext, args: Dict[str, Any]):
    date_format = '%Y-%m-%d'

    aggregated_prices_dir, aggregated_prices_version_name = ypath_split(args['aggregated_prices_path'])
    date_parsed = datetime.datetime.strptime(aggregated_prices_version_name, date_format)

    current_version_path = ytlib.join(ctx.current_dataset_path, aggregated_prices_version_name)

    ctx.yt_client.create('map_node', current_version_path, recursive=True)
    num_tables_for_aggregation = int(args['num_tables_for_aggregation'])

    ctx.run_yql_builder(
        'prices_and_availabilities/build_cluster_permalink_availabilities.yql',
        yql_params={
            '$cluster_permalink_per_night_prices_dir': aggregated_prices_dir,
            '$cluster_permalink_per_night_prices_from': datetime.datetime.strftime(date_parsed - datetime.timedelta(num_tables_for_aggregation - 1), date_format),
            '$cluster_permalink_per_night_prices_to': datetime.datetime.strftime(date_parsed, date_format),
            '$output_path': ytlib.join(current_version_path, 'cluster_permalink_availabilities'),
            '$ema_ndays': num_tables_for_aggregation,
        }
    )

    ctx.run_yql_builder(
        'prices_and_availabilities/build_hotel_id_availabilities.yql',
        yql_params={
            '$hotel_id_per_night_prices_dir': aggregated_prices_dir,
            '$hotel_id_per_night_prices_from': datetime.datetime.strftime(date_parsed - datetime.timedelta(num_tables_for_aggregation - 1), date_format),
            '$hotel_id_per_night_prices_to': datetime.datetime.strftime(date_parsed, date_format),
            '$output_path': ytlib.join(current_version_path, 'hotel_id_availabilities'),
            '$ema_ndays': num_tables_for_aggregation,
        }
    )
