# -*- coding: utf-8 -*-
import logging
from typing import Dict

import yaml
from travel.library.python.arcadia.arcadia_tools import get_arcadia_path

from travel.hotels.tools.dataset_curator.data import Dataset, ENV_SUFFIXES

LOG = logging.getLogger(__name__)


class SbPlannerPlanBuilder:
    @staticmethod
    def _repr_str(dumper, data):
        if '\n' in data:
            return dumper.represent_scalar(u'tag:yaml.org,2002:str', data, style='|')
        return dumper.represent_scalar(u'tag:yaml.org,2002:str', data, style=None)

    def update_plan(self, datasets: Dict[str, Dataset]):
        plan = {}

        for name, dataset in datasets.items():
            if dataset.build is not None and dataset.build.sb_planner is not None:
                sb_planner = dataset.build.sb_planner
                plan_item_name = f'dataset_curator_build_{name}'

                args = ['build',
                        '--yt-token {{yt_token}}',
                        '--env {{env}}',
                        '--aux-data-yt-dir //home/travel/{{env}}/dataset_curator/aux_data',
                        '--yql-token {{yql_token}}',
                        '--tmp-dir //home/travel/{{env}}/dataset_curator/building/tmp',
                        f'--dataset {name}']

                args_per_env = {
                    '': args
                }

                if len(sb_planner.builder_args_per_suffix) > 0:
                    args += ['--builder-args']

                    for suffix, builder_args in sb_planner.builder_args_per_suffix.items():
                        if suffix not in args_per_env:
                            args_per_env[suffix] = []
                        args_per_env[suffix] += [
                            f'{k}={v}'
                            for k, v in builder_args.items()
                        ]

                plan_item = {
                    'tags': ['PRJ:HOTELS', f'OWNER:{dataset.owner}'],
                    'resource': 'TRAVEL_HOTELS_DATASET_CURATOR',
                    'args': '\n'.join(args)
                }
                for suffix, curr_args in args_per_env.items():
                    if suffix in ENV_SUFFIXES:
                        plan_item[f'args{suffix}'] = '\n'.join(curr_args)
                for k, v in sb_planner.sb_planner_args.items():
                    plan_item[k] = v

                plan[plan_item_name] = plan_item

        yaml.SafeDumper.add_representer(str, SbPlannerPlanBuilder._repr_str)
        plan_data = yaml.safe_dump(plan)
        for suffix in [''] + ENV_SUFFIXES:
            plan_data = plan_data.replace(f'args{suffix}: |-', f'args{suffix}: >')

        path = get_arcadia_path('travel', 'hotels', 'devops', 'sandbox_planner', 'plan',  'dataset_curator.yaml')
        LOG.info(f'Saving plan to: {path}')
        with open(path, 'w') as f:
            f.write(plan_data)
