# -*- coding: utf-8 -*-

import logging
from dataclasses import dataclass
from typing import Dict, List, Any, Optional

import yt

LOG = logging.getLogger(__name__)


@dataclass
class BaseTypeSchema:
    nullable: bool
    description: Optional[str]

    def validate_value(self, value: Any) -> List[str]:
        if value is None:
            if not self.nullable:
                return ['Expected not-null, but null found']
            return []

        return self._validate_specific_value(value)

    def _validate_specific_value(self, value: Any) -> List[str]:
        raise NotImplementedError()


@dataclass
class PrimitiveTypeSchema(BaseTypeSchema):
    value_type: str
    Mapping = {
        'int32': int,  # TODO: add bounds validation and maybe yt schema validation
        'uint32': int,
        'int64': int,
        'uint64': yt.yson.yson_types.YsonUint64,
        'boolean': bool,
        'double': float,
    }

    def _validate_specific_value(self, value: Any) -> List[str]:
        if not self._is_matching_type(type(value)):
            return [f'Type "{type(value)}" is not matching for expected "{self.value_type}"']
        return []

    def _is_matching_type(self, data_type: type):
        return self.Mapping[self.value_type] == data_type

    @staticmethod
    def get_supported_type_names():
        return list(PrimitiveTypeSchema.Mapping.keys())


@dataclass
class StringTypeSchema(BaseTypeSchema):
    values: List[str]

    def _validate_specific_value(self, value: Any) -> List[str]:
        if type(value) != str:
            return [f'Expected string, but got {type(value)}']
        if self.values is not None:
            if value not in self.values:
                return [f'Expected one of {self.values}, but got {value}']
        return []


@dataclass
class ListTypeSchema(BaseTypeSchema):
    elements: BaseTypeSchema

    def _validate_specific_value(self, value: Any) -> List[str]:
        if type(value) != list:
            return [f'Expected list, but got {type(value)}']
        res = []
        for element in value:
            res += self.elements.validate_value(element)
        return res


@dataclass
class DictTypeSchema(BaseTypeSchema):
    keys: BaseTypeSchema
    values: BaseTypeSchema

    def _validate_specific_value(self, value: Any) -> List[str]:
        if type(value) != dict:
            return [f'Expected dict, but got {type(value)}']
        res = []
        for k, v in value.items():
            if self.keys is not None:
                res += self.keys.validate_value(k)
            if self.values is not None:
                res += self.values.validate_value(v)
        return res


@dataclass
class StructTypeSchema(BaseTypeSchema):
    allow_unknown_fields: str
    fields: Dict[str, BaseTypeSchema]

    def _validate_specific_value(self, value: Any) -> List[str]:
        if type(value) != dict:
            return [f'Expected struct (as dict), but got {type(value)}']

        res = []
        if not self.allow_unknown_fields:
            for name in value:
                if name not in self.fields:
                    res.append(f'Field {name} is unknown')
        for name, field_type in self.fields.items():
            for err in field_type.validate_value(value.get(name)):
                res.append(f'Field {name}: ' + err)
            # TODO: distinguish field absence and null field
        return res


@dataclass
class AnyTypeSchema(BaseTypeSchema):
    def _validate_specific_value(self, value: Any) -> List[str]:
        return []
