# -*- coding: utf-8 -*-

import json
from typing import Dict, Tuple, List, Any


class ValidationError:
    def __init__(self, error_name: str):
        self.error_name = error_name

    def merge(self, other: 'ValidationError'):
        if type(self) != type(other):
            raise ValueError('Can\'t merge errors of different types')
        if self.error_name != other.error_name:
            raise ValueError('Can\'t merge errors with different error_name')
        return self._do_merge(other)

    def dump_as_dict(self):
        raise NotImplementedError()

    def _do_merge(self, other):
        raise NotImplementedError()


class ValidationErrorWithSamples(ValidationError):
    MaxSamples = 3

    def __init__(self, error_name: str, count: int, samples: List[Any]):
        super().__init__(error_name)
        self.count = count
        self.samples = samples[:self.MaxSamples]

    def __str__(self):
        return f'{self.error_name} ({self.count} times), samples: {json.dumps(self.samples)[:100]} (trimmed to 100 chars)'

    def dump_as_dict(self):
        return {
            'error_name': self.error_name,
            'count': self.count,
            'samples': self.samples
        }

    def _do_merge(self, other: 'ValidationErrorWithSamples'):
        return ValidationErrorWithSamples(self.error_name, self.count + other.count, (self.samples + other.samples)[:self.MaxSamples])


class ValidationErrorWithCounts(ValidationError):
    def __init__(self, error_name: str, count: int):
        super().__init__(error_name)
        self.count = count

    def __str__(self):
        return f'{self.error_name} ({self.count} times)'

    def dump_as_dict(self):
        return {
            'error_name': self.error_name,
            'count': self.count
        }

    def _do_merge(self, other: 'ValidationErrorWithCounts'):
        return ValidationErrorWithCounts(self.error_name, self.count + other.count)


class ValidationResult:
    def __init__(self):
        self.error_cnt = 0
        self.errors: Dict[Tuple[type, str], ValidationError] = {}

    def __str__(self):
        if self.is_ok():
            return 'OK'
        errors = [' ' * 4 + str(x) for x in self.errors.values()]
        return f'Errors ({len(errors)}):\n' + '\n'.join(errors)

    def is_ok(self):
        return len(self.errors) == 0

    def add_error(self, error: ValidationError):
        self.error_cnt += 1
        key = (type(error), error.error_name)
        if key not in self.errors:
            self.errors[key] = error
        else:
            self.errors[key] = self.errors[key].merge(error)

    def add_errors_from(self, other: 'ValidationResult'):
        for err in other.errors.values():
            self.add_error(err)
