import logging
import sys
import os
from argparse import ArgumentParser

import numpy as np
import pandas as pd
from startrek_client import Startrek
from travel.hotels.lib.python3.cli.cli import auto_progress_reporter
from travel.hotels.lib.python3.yt import ytlib
from travel.library.python.tools import replace_args_from_env
from yt.wrapper import YtClient


class Runner(object):
    def __init__(self, args):
        self.yt_client = YtClient(proxy=args.yt_proxy, token=args.yt_token)
        self.st_client = Startrek(useragent='python', base_url='https://st-api.yandex-team.ru', token=args.st_token)
        self.expedia_statements_path, self.expedia_statement_name = ytlib.ypath_split(args.expedia_current_statement_path)
        self.args = args

    def get_expedia_orders_df(self):
        orders = {}
        all_tables = list(self.yt_client.list(self.expedia_statements_path, absolute=True))
        for table_name in auto_progress_reporter(all_tables, total=len(all_tables), name='Load expedia statements'):
            for row in self.yt_client.read_table(table_name):
                order_id = row['Affiliate Reference Number']
                if order_id not in orders:
                    orders[order_id] = {'max_book_amount': 0}
                if row['Transaction Type'] == 'Book':
                    # Seems like order_amount in cpa is the max amount: refunds don't affect order_amount, but additional payments do
                    orders[order_id]['max_book_amount'] = max(orders[order_id]['max_book_amount'], row['Amount'])

        return pd.DataFrame.from_dict(data=orders, orient='index')

    def get_yndx_orders_df(self):
        yndx_orders = {}

        cols = ['order_amount', 'discount_amount', 'yandex_plus_withdraw_points']
        cpa_path = self.yt_client.TablePath(self.args.cpa_orders_path, columns=cols + ['partner_name', 'partner_order_id'])
        for row in auto_progress_reporter(self.yt_client.read_table(cpa_path), total=self.yt_client.row_count(cpa_path), name='Load expedia orders from cpa'):
            if row['partner_name'] != 'expedia':
                continue
            order_id = row['partner_order_id']
            if not order_id.endswith(':0'):
                order_id = order_id + ':0'
            yndx_orders[order_id] = {k: v for k, v in row.items() if k in cols}

        return pd.DataFrame.from_dict(data=yndx_orders, orient='index')

    def get_joined_orders_df(self):
        expedia_orders_df = self.get_expedia_orders_df()
        yndx_orders_df = self.get_yndx_orders_df()

        joined = pd.merge(expedia_orders_df, yndx_orders_df, how='inner', left_index=True, right_index=True)
        joined['usd_to_rub'] = joined.apply(lambda row: row["order_amount"] / row["max_book_amount"] if row["max_book_amount"] > 0 else np.nan, axis=1)
        joined['discount_amount_usd'] = joined.apply(lambda row: row["discount_amount"] / row["usd_to_rub"], axis=1)
        joined['yandex_plus_withdraw_points_usd'] = joined.apply(lambda row: row["yandex_plus_withdraw_points"] / row["usd_to_rub"], axis=1)

        return joined

    def validate_discount_amounts(self, joined_orders_df, current_expedia_statement):
        joined_current = joined_orders_df.loc[list(current_expedia_statement['Affiliate Reference Number'])]

        orders_with_discount = set(joined_current[(joined_current['discount_amount'] > 0) | (joined_current['yandex_plus_withdraw_points'] > 0)].index)
        # We check only orders with discount in some validations to avoid failing on irrelevant orders

        mean_rub_to_usd = joined_current[~joined_current['usd_to_rub'].isna()]['usd_to_rub'].mean()
        logging.info(f'Mean fallback FX rate: {mean_rub_to_usd}')

        orders_with_non_finite_usd_to_rub = orders_with_discount & set(joined_current.loc[~np.isfinite(joined_current['usd_to_rub'])].index)
        assert len(orders_with_non_finite_usd_to_rub) == 0, f'There are orders with non-finite usd rate: {orders_with_non_finite_usd_to_rub}'
        # non-finite numbers are possible only for irrelevant orders (because of assert), so setting them to zero
        joined_current.loc[~np.isfinite(joined_current['usd_to_rub']), 'usd_to_rub'] = 0

        orders_with_outstanding_usd_to_rub = joined_current[(joined_current['usd_to_rub'] < 0.9 * mean_rub_to_usd) | (1.1 * mean_rub_to_usd < joined_current['usd_to_rub'])]['usd_to_rub'].to_dict()
        orders_with_outstanding_usd_to_rub = {k: v for k, v in orders_with_outstanding_usd_to_rub.items() if k in orders_with_discount}
        assert len(orders_with_outstanding_usd_to_rub) == 0, f'There are orders with suspicious usd rate: {orders_with_outstanding_usd_to_rub}'

        orders_with_negative_discount_amount_usd = joined_current[joined_current['discount_amount_usd'] < 0]['discount_amount_usd'].to_dict()
        assert len(orders_with_negative_discount_amount_usd) == 0, "There are orders with negative discount_amount_usd: {}"

        orders_with_negative_discount_amount_usd = joined_current[joined_current['yandex_plus_withdraw_points_usd'] < 0]['yandex_plus_withdraw_points_usd'].to_dict()
        assert len(orders_with_negative_discount_amount_usd) == 0, "There are orders with negative yandex_plus_withdraw_points_usd: {}"

    def build_marketing_expenses(self, joined_orders_df, current_expedia_statement, column_name):
        discount_amount_usd_dict = joined_orders_df.to_dict()[column_name]

        def calc_expenses(row):
            discount = discount_amount_usd_dict[row['Affiliate Reference Number']]
            if discount > 0:
                discount = round(discount, 2)
                if row["Transaction Type"] == "Book":
                    return discount
                elif row["Transaction Type"] == "Cancel":
                    return -discount
            return 0

        return pd.to_numeric(current_expedia_statement.apply(calc_expenses, axis=1))

    def post_comment_with_results(self, statement_name, total_marketing_expenses, total_yandex_plus_expenses, marketing_expenses_file):
        restored_statement_name = 'Yandex ' + statement_name[:-len('_Statement')]
        comment_text = f'''Отчет по маркетинговым расходам для Expedia по файлу %%{restored_statement_name}%%
Сумма маркетинговых расходов: %%{total_marketing_expenses:.2f} USD%%
Сумма списаний баллов плюса: %%{total_yandex_plus_expenses:.2f} USD%%
В приложенном файле добавлены колонки:
* %%Marketing Expenses%%, содержащая сумму маркетинговых расходов в рамках транзакции.
* %%Yandex Plus Expenses%%, содержащая сумму списаний баллов плюса в рамках транзакции.
'''

        self.st_client.issues[self.args.st_ticket].comments.create(text=comment_text, attachments=[marketing_expenses_file])

    def run(self):
        marketing_expenses_file = 'marketing_expenses.xlsx'

        joined_orders_df = self.get_joined_orders_df()
        current_expedia_statement = pd.DataFrame(data=self.yt_client.read_table(ytlib.join(self.expedia_statements_path, self.expedia_statement_name)))
        if not self.args.skip_validation:
            self.validate_discount_amounts(joined_orders_df, current_expedia_statement)
        current_expedia_statement["Marketing Expenses"] = self.build_marketing_expenses(joined_orders_df, current_expedia_statement, 'discount_amount_usd')
        current_expedia_statement["Yandex Plus Expenses"] = self.build_marketing_expenses(joined_orders_df, current_expedia_statement, 'yandex_plus_withdraw_points_usd')
        current_expedia_statement.to_excel(marketing_expenses_file)

        self.post_comment_with_results(self.expedia_statement_name, current_expedia_statement["Marketing Expenses"].sum(),
                                       current_expedia_statement["Yandex Plus Expenses"].sum(), marketing_expenses_file)
        os.remove(marketing_expenses_file)


def main():
    logging.basicConfig(level=logging.INFO, format="%(asctime)-15s | %(module)s | %(levelname)s | %(message)s", stream=sys.stdout)
    logging.getLogger('yt.packages.urllib3.connectionpool').setLevel(logging.WARNING)

    parser = ArgumentParser()
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--st-token', required=True)
    parser.add_argument('--st-ticket', required=True)
    parser.add_argument('--expedia-current-statement-path', required=True)
    parser.add_argument('--cpa-orders-path', default='//home/travel/prod/cpa/hotels/orders')
    parser.add_argument('--skip-validation', type=bool, default=False)
    args = parser.parse_args(args=replace_args_from_env())
    Runner(args).run()


if __name__ == '__main__':
    main()
