
import logging
import re
import sys
from argparse import ArgumentParser

from yt.wrapper import YtClient
import yt.wrapper as yt

from parse import parse
from travel.hotels.lib.python3.yt import ytlib
from travel.library.python.tools import replace_args_from_env

LOG = logging.getLogger(__name__)

LOG_SCHEMA_FIELDS = {
    'Itinerary': 'string',
    'GBV': 'double',
    'GBV (Rub)': 'double',
    'Estimated Fee': 'double',
    'Event Type': 'string',
    'Source Table Type': 'string',
    'Amount Payable': 'double'
}


class Reader:
    def __init__(self, args):
        self.args = args
        self.log_path = yt.ypath_join(args.yt_path, 'ExpediaBookingsEventLog')
        self.yt_client = YtClient(proxy=args.yt_proxy, token=args.yt_token)

    def read_table(self, yt_path, table_type):
        with self.yt_client.Transaction():
            if not self.check_table_is_read(yt_path):
                LOG.info("Processing {}: {}".format(table_type, yt_path))
                rows = self.yt_client.read_table(yt_path)
                log = []
                for row in rows:
                    itinerary_id = row['Itinerary Number']
                    gbv = row['Gross Booking Amount (GBV) USD']
                    gbv_rub = row['Gross Booking Amount (Local Currency)']
                    event_type = row['Room Booking Status']
                    estimated_fee = row['Estimated Marketing Fee USD']
                    log_row_data = [itinerary_id, gbv, gbv_rub, estimated_fee, event_type, table_type, 0.0]
                    log_row = dict(zip(LOG_SCHEMA_FIELDS.keys(), log_row_data))
                    log.append(log_row)
                self.yt_client.write_table(self.yt_client.TablePath(self.log_path, append=True), log)
                self.mark_table_read(yt_path)

    def read_statements_table(self, yt_path):
        with self.yt_client.Transaction():
            if not self.check_table_is_read(yt_path):
                LOG.info("Processing statements: {}".format(yt_path))
                rows = self.yt_client.read_table(yt_path)
                log = []
                for row in rows:
                    itinerary_id = row['EAN Itinerary ID']
                    amount = row['Amount']
                    amount_payable = row['Amount Payable']
                    event_type = row['Transaction Type']
                    fee = row['Marketing Fee Amount']
                    log_row_data = [itinerary_id, amount, 0.0, fee, event_type, 'statements', amount_payable]
                    log_row = dict(zip(LOG_SCHEMA_FIELDS.keys(), log_row_data))
                    log.append(log_row)
                self.yt_client.write_table(self.yt_client.TablePath(self.log_path, append=True), log)
                self.mark_table_read(yt_path)

    def mark_table_read(self, yt_path):
        self.yt_client.set_attribute(path=yt_path, attribute='processed', value=True)

    def check_table_is_read(self, yt_path) -> bool:
        return self.yt_client.get_attribute(path=yt_path, attribute='processed', default=False)

    def ensure_log_table_exists(self):
        table_path = yt.ypath_join(self.args.yt_path, 'ExpediaBookingsEventLog')
        ytlib.ensure_table_exists(table_path, self.yt_client, ytlib.schema_from_dict(LOG_SCHEMA_FIELDS))
        self.yt_client.set_attribute(path=table_path, attribute='optimize_for', value='scan')

    def read_booked(self):
        booked_path = yt.ypath_join(self.args.yt_path, 'booked')
        for name in self.yt_client.list(booked_path):
            if re.match(r".*EstimatedCommissions$", name):
                self.read_table(yt.ypath_join(booked_path, name), 'booked')

    def read_stayed(self):
        stayed_path = yt.ypath_join(self.args.yt_path, 'stayed')
        for name in self.yt_client.list(stayed_path):
            if re.match(r".*EstimatedCommissions$", name):
                self.read_table(yt.ypath_join(stayed_path, name), 'stayed')

    def read_statements(self):
        statements_path = yt.ypath_join(self.args.yt_path, 'statements')
        for name in self.yt_client.list(statements_path):
            self.read_statements_table(yt.ypath_join(statements_path, name))


def main():
    logging.basicConfig(level=logging.INFO, format="%(asctime)-15s | %(module)s | %(levelname)s | %(message)s",
                        stream=sys.stdout)
    logging.getLogger('yt.packages.urllib3.connectionpool').setLevel(logging.WARNING)
    parser = ArgumentParser()
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--yt-path', default='//home/travel/alexcrush/expedia_bookings/mail_files')
    args = parser.parse_args(args=replace_args_from_env())

    LOG.info("Start processing")
    reader = Reader(args)
    reader.ensure_log_table_exists()
    reader.read_booked()
    reader.read_stayed()
    reader.read_statements()


def parse_datetime_iso(dt_text):
    return parse('{:ti}', dt_text)[0]


if __name__ == '__main__':
    main()
