import datetime
import logging
import sys
from argparse import ArgumentParser

from yt.wrapper import YtClient
import yt.wrapper as yt

from parse import parse
from travel.hotels.lib.python3.yt import ytlib
from travel.library.python.tools import replace_args_from_env

LOG = logging.getLogger(__name__)

LOG_SCHEMA_FIELDS = {
    'Itinerary': 'string',
    'GBV': 'double',
    'GBV (Rub)': 'double',
    'Estimated Fee': 'double',
    'Event Type': 'string',
    'Source Table Type': 'string',
    'Amount Payable': 'double'
}

AGGREGATE_FIELDS = {
    'Itinerary': 'string',
    'Initial Amount': 'double',
    'Current Amount': 'double',
    'Fee': 'double',
    'UpdatedAt': 'string',
    'State': 'string',
}


class Parser:
    def __init__(self, args):
        self.args = args
        self.log_path = yt.ypath_join(args.yt_path, 'ExpediaBookingsEventLog')
        self.aggregate_path = yt.ypath_join(args.yt_path, 'ExpediaItineraryData')
        self.yt_client = YtClient(proxy=args.yt_proxy, token=args.yt_token)

    def parse_booked_log_rows(self, rows):
        gbv = 0.0
        fee = 0.0
        cur = 0.0
        status = 'New'
        for row in rows:
            row_type = row['Event Type']
            if row_type == 'Purchased':
                gbv = row['GBV']
                status = 'Purchased'
            cur = cur + row['GBV']
            fee = fee + row['Estimated Fee']
        cancelled = list(filter(lambda it: it['Event Type'] == 'Cancelled', rows))
        if len(cancelled) > 0:
            row = cancelled[0]
            cur = row['GBV']
            fee = row['Estimated Fee']
            status = 'Cancelled'

        return gbv, cur, fee, status

    def parse_stmt_log_rows(self, rows):
        gbv = 0.0
        cur = 0.0
        fee = 0.0
        status = 'Purchased'
        for stmt in rows:
            if stmt['Event Type'] == 'Cancel':
                status = 'Cancelled'
            if stmt['Event Type'] == 'Book':
                gbv = stmt['GBV']
            cur = cur + stmt['GBV']
            fee = fee + stmt['Estimated Fee']
        return gbv, cur, fee, status

    def read_log(self):
        with self.yt_client.Transaction():
            LOG.info("Reading log")
            rows = self.yt_client.read_table(self.log_path)
            log_by_itno = dict()
            for row in rows:
                itno_item = log_by_itno.get(row['Itinerary'], [])
                itno_item.append(row)
                log_by_itno[row['Itinerary']] = itno_item
            aggregates = []
            for itno, rows in log_by_itno.items():
                stmt_rows = list(filter(lambda it: it['Source Table Type'] == 'statements', rows))
                stay_rows = list(filter(lambda it: it['Source Table Type'] == 'stayed', rows))
                book_rows = list(filter(lambda it: it['Source Table Type'] == 'booked', rows))
                gbv = 0.0
                fee = 0.0
                status = 'Purchased'
                if len(book_rows) > 0:
                    LOG.info('Booked for {}'.format(itno))
                    gbv, cur, fee, status = self.parse_booked_log_rows(book_rows)
                elif len(stay_rows) > 0:
                    LOG.info('Stayed for {}'.format(itno))
                    gbv, cur, fee, status = self.parse_booked_log_rows(stay_rows)
                if len(stmt_rows) > 0:
                    LOG.info('Statements for {}'.format(itno))
                    stmt_gbv, cur, fee, status = self.parse_stmt_log_rows(stmt_rows)
                    if stmt_gbv != gbv:
                        LOG.warn("GBV from statement differs: {} - {}".format(stmt_gbv, gbv))
                aggregates.append(
                    dict(zip(AGGREGATE_FIELDS.keys(), [itno, gbv, cur, fee, datetime.date.today().isoformat(), status])))
            LOG.info("Processing done. Writing aggregates...")
            self.yt_client.write_table(self.yt_client.TablePath(self.aggregate_path, append=False), aggregates)

    def ensure_log_table_exists(self):
        ytlib.ensure_table_exists(self.log_path, self.yt_client, ytlib.schema_from_dict(LOG_SCHEMA_FIELDS))

    def ensure_aggregate_table_exists(self):
        ytlib.ensure_table_exists(self.aggregate_path, self.yt_client, ytlib.schema_from_dict(AGGREGATE_FIELDS))


def main():
    logging.basicConfig(level=logging.INFO, format="%(asctime)-15s | %(module)s | %(levelname)s | %(message)s",
                        stream=sys.stdout)
    logging.getLogger('yt.packages.urllib3.connectionpool').setLevel(logging.WARNING)
    parser = ArgumentParser()
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--yt-path', default='//home/travel/akormushkin/expedia_bookings/mail_files')
    args = parser.parse_args(args=replace_args_from_env())

    LOG.info("Start processing")
    reader = Parser(args)
    reader.ensure_log_table_exists()
    reader.ensure_aggregate_table_exists()
    reader.read_log()


def parse_datetime_iso(dt_text):
    return parse('{:ti}', dt_text)[0]


if __name__ == '__main__':
    main()
