# -*- coding: utf-8 -*-

from argparse import ArgumentParser
import logging
import sys

from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient

from travel.library.python.tools import replace_args_from_env

from travel.hotels.lib.python3.yql import yqllib
from travel.hotels.lib.python3.yt.versioned_path import VersionedPath
from travel.hotels.lib.python3.yt import ytlib

LOG = logging.getLogger(__name__)


class Runner:
    def __init__(self, args):
        self.yql_client = YqlClient(db=args.yt_proxy, token=args.yql_token)
        self.yt_client = YtClient(proxy=args.yt_proxy, token=args.yt_token)
        self.args = args

    def run(self):
        versioned_path = VersionedPath(self.args.yt_path, yt_client=self.yt_client)
        with versioned_path as work_path:
            self._run_yql(work_path)
            # copy results to another cluster
            if self.args.export_results:
                versioned_path.transfer_results(self.args.export_destination, self.args.yt_token, self.args.yt_proxy)

    def _run_yql(self, work_path):
        property_pansions = ytlib.join(work_path, 'property_pansions')

        yqllib.run_yql_file(
            self.yql_client,
            'build_pansions.yql',
            'Travel',
            'ExpediaPropertyPansionsBuilder',
            parameters={
                '$pansions_table_path': property_pansions,
            }
        )


def main():
    logging.basicConfig(level=logging.INFO, format="%(asctime)-15s | %(module)s | %(levelname)s | %(message)s",
                        stream=sys.stdout)
    logging.getLogger('yt.packages.urllib3.connectionpool').setLevel(logging.WARNING)
    parser = ArgumentParser()
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--yql-token', required=True)
    parser.add_argument('--yt-path', required=True)
    parser.add_argument("--export-results", action='store_true')
    parser.add_argument("--export-destination", default='arnold')
    args = parser.parse_args(args=replace_args_from_env())
    Runner(args).run()


if __name__ == '__main__':
    main()

