#include "travel/library/proto/logging_ydb/logging_ydb.pb.h"
#include "travel/hotels/offercache/proto/reqans_logrecord.pb.h"
#include "travel/hotels/offercache/proto/grpc_reqans_logrecord.pb.h"
#include "travel/hotels/offercache/proto/cacheusage_logrecord.pb.h"
#include "travel/hotels/pricechecker/proto/reqans_logrecord.pb.h"
#include <travel/hotels/pricechecker/proto/pricechecker_logrecord.pb.h>
#include "travel/hotels/redir/proto/reqans_hotels.pb.h"
#include "travel/hotels/redir/proto/reqans_trains.pb.h"
#include "travel/hotels/redir/proto/reqans_suburban.pb.h"
#include "travel/hotels/lib/cpp/grpc/reqans_logrecords.pb.h"
#include "travel/orders/proto/services/orders/cpa/cpa.pb.h"
#include "travel/api/proto/happy_page/happy_page.pb.h"
#include "travel/api/proto/hotels_portal/hotels_suggest.pb.h"

#include "travel/hotels/tools/gen_logfeller_parser/parser.pb.h"
#include <travel/hotels/lib/cpp/util/arcadia.h>

#include <library/cpp/getopt/opt.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <util/generic/map.h>
#include <util/generic/vector.h>
#include <util/generic/string.h>
#include <util/folder/path.h>
#include <util/stream/file.h>
#include <util/string/subst.h>

const char g_NameDelim = '_';
const char g_PathDelim = '/';

struct TLogDescription {
    THolder<NProtoBuf::Message> Message;
    TString ConfigFile;
};

class TGenerator {
public:
    void AddFields(const NProtoBuf::Message& proto, const TString& namePfx, const TString& pathPfx, TParser* parserProto) {
        auto d = proto.GetDescriptor();

        for (int i = 0, end = d->field_count(); i < end; ++i) {
            const google::protobuf::FieldDescriptor* fd = d->field(i);
            auto name = fd->options().HasExtension(NTravelProto::LogName) ? fd->options().GetExtension(NTravelProto::LogName) : fd->name();
            bool is_any = fd->is_repeated() || (fd->message_type() && fd->message_type()->full_name() == google::protobuf::Any::descriptor()->full_name());

            if (fd->cpp_type() == google::protobuf::FieldDescriptor::CPPTYPE_MESSAGE && !is_any) {
                // And we need to go deeper
                auto skipNameInLog = fd->options().HasExtension(NTravelProto::ExtractNestedFieldsInLog) && fd->options().GetExtension(NTravelProto::ExtractNestedFieldsInLog);
                AddFields(proto.GetReflection()->GetMessage(proto, fd),
                          skipNameInLog ? namePfx : namePfx + name + g_NameDelim,
                          skipNameInLog ? pathPfx : pathPfx + name + g_PathDelim,
                          parserProto
                );
                continue;
            }
            TParser::TField::EFieldType ft;

            if (is_any) {
                ft = TParser::TField::VT_ANY;
            } else {
                switch (fd->cpp_type()) {
                    case google::protobuf::FieldDescriptor::CPPTYPE_INT32:
                        ft = TParser::TField::VT_INT64;
                        break;
                    case google::protobuf::FieldDescriptor::CPPTYPE_INT64:
                        ft = TParser::TField::VT_INT64;
                        break;

                    case google::protobuf::FieldDescriptor::CPPTYPE_UINT32:
                        ft = TParser::TField::VT_UINT64;
                        break;
                    case google::protobuf::FieldDescriptor::CPPTYPE_UINT64:
                        ft = TParser::TField::VT_UINT64;
                        break;

                    case google::protobuf::FieldDescriptor::CPPTYPE_DOUBLE:
                    case google::protobuf::FieldDescriptor::CPPTYPE_FLOAT:
                        ft = TParser::TField::VT_DOUBLE;
                        break;

                    case google::protobuf::FieldDescriptor::CPPTYPE_BOOL:
                        ft = TParser::TField::VT_BOOLEAN;
                        break;

                    case google::protobuf::FieldDescriptor::CPPTYPE_ENUM:
                        ft = TParser::TField::VT_STRING;
                        break;

                    case google::protobuf::FieldDescriptor::CPPTYPE_STRING:
                        ft = TParser::TField::VT_STRING;
                        break;

                    case google::protobuf::FieldDescriptor::CPPTYPE_MESSAGE:
                        Y_VERIFY(false);// already handled above
                };
            }
            auto pf = parserProto->Addfields();
            pf->set_name(namePfx + name);
            pf->set_path(pathPfx + name);
            pf->set_type(ft);
        }
    }
};


int main(int argc, const char* argv[]) {
    TString logName;
    {
        NLastGetopt::TOpts opts;
        opts.AddHelpOption('h');
        opts.AddLongOption('l', "log", "Log name")
            .StoreResult(&logName);
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);
    }

    TMap<TString, TLogDescription> logFactory;
    logFactory.emplace("offercache-reqans", TLogDescription{
                               THolder(new NTravelProto::NOfferCache::NReqAnsLog::TReqAnsLogRecord()),
                               "travel-hotels-offercache-log.json"
                           });
    logFactory.emplace("offercache-grpc-reqans", TLogDescription{
                               THolder(new NTravelProto::NOfferCache::NGrpcReqAnsLog::TGrpcReqAnsLogRecord()),
                               "travel-hotels-offercache-grpc-reqans-log.json"
                           });
    logFactory.emplace("offercache-cacheusage", TLogDescription{
                               THolder(new NTravelProto::NOfferCache::TCacheUsageLogRecord()),
                               "travel-hotels-offercache-cacheusage-log.json"
                           });
    logFactory.emplace("pricechecker-old", TLogDescription{
                               THolder(new NTravelProto::NPriceChecker::TReqAnsLogRecord()),
                               "travel-hotels-pricechecker-log.json"
                           });
    logFactory.emplace("pricechecker", TLogDescription{
                               THolder(new NTravelProto::NPriceChecker::TPriceCheckerLogRecord()),
                               "travel-hotels-pricechecker-results.json"
                           });
    logFactory.emplace("redir-hotels", TLogDescription{
                               THolder(new NTravelProto::NRedir::NHotels::TReqAnsLogRecord()),
                               "travel-redir-log.json"
                           });
    logFactory.emplace("redir-trains", TLogDescription{
                               THolder(new NTravelProto::NRedir::NTrains::TReqAnsLogRecord()),
                               "travel-redir-trains-log.json"
                           });
    logFactory.emplace("redir-suburban", TLogDescription{
                               THolder(new NTravelProto::NRedir::NSuburban::TReqAnsLogRecord()),
                               "travel-redir-suburban-log.json"
                           });
    logFactory.emplace("train-refunds-cpa-export", TLogDescription{
                               THolder(new NTravelProto::NOrders::NCpaExport::TTrainRefundExportLogRecord()),
                               "travel-train-refunds-cpa-export-log.json"
                           });
    logFactory.emplace("orders-ydb-log-copy", TLogDescription{
                               THolder(new NTravelProto::NOrders::NYdbLog::TOrderLogRecord()),
                               "travel-orders-ydb-log-copy.json"
                           });
    logFactory.emplace("happy-page-log", TLogDescription{
                               THolder(new NTravelProto::NApi::NHappyPage::THappyPageLogRecord()),
                               "travel-happy-page-log.json"
                           });
    logFactory.emplace("hotels-suggest-log", TLogDescription{
                               THolder(new NTravelProto::NApi::NHotelsPortal::THotelsSuggestLogRecord()),
                               "travel-api-hotels-suggest-log.json"
                           });
    logFactory.emplace("hotels-suggest-choice-log", TLogDescription{
                               THolder(new NTravelProto::NApi::NHotelsPortal::THotelsSuggestChoiceLogRecord()),
                               "travel-api-hotels-suggest-choice-log.json"
                           });
    logFactory.emplace("hotels-geocounter-reqans-log", TLogDescription{
                               THolder(new NTravelProto::NGrpcReqAnsLog::TReqAnsLogRecord()),
                               "travel-hotels-geocounter-grpc-reqans-log.json"
                           });

    const auto iter = logFactory.find(logName);
    if (iter == logFactory.end()) {
        Cerr << "The log name is not supported, please choose one of the following:" << Endl;
        for (const auto& element : logFactory) {
            Cerr << "  " << element.first << Endl;
        }
        return 1;
    }
    const auto& logDescription = iter->second;

    TParser proto;
    TGenerator().AddFields(*logDescription.Message, "", "", &proto);

    auto cfg = NProtobufJson::TProto2JsonConfig()
                    .SetFormatOutput(true)
                    .SetMissingRepeatedKeyMode(NProtobufJson::TProto2JsonConfig::MissingKeyDefault)
                    .SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName);
    TString json = NProtobufJson::Proto2Json(proto, cfg);

    TFsPath configPath = NTravel::GetArcadiaPath({"logfeller", "configs", "parsers", logDescription.ConfigFile});

    TOFStream(configPath).Write(json);
    Cerr << "New config is written to " << configPath << " , don't forget to commit it!" << Endl;
    return 0;
}

